<?php
if (!defined('_PS_VERSION_')) {
  exit;
}

/**
 * Client per le chiamate API verso Poinzilla
 * Gestisce tutte le comunicazioni HTTP con l'API di Poinzilla
 */
class PoinzillaApiClient
{
  /** @var string Header per l'autenticazione */
  private const AUTH_HEADER = 'X-loyalty-channel-key';

  /**
   * Esegue una richiesta HTTP verso gli endpoint di Poinzilla
   *
   * @param string $url URL dell'endpoint
   * @param array $data Dati da inviare nel body della richiesta
   * @return array Array contenente ['response' => string, 'code' => int, 'error' => string]
   */
  public function poinzillaHTTPClient($url, $data)
  {
    PoinzillaLogger::log("Inizializzazione chiamata API a: " . $url, 1);

    $curl = curl_init();

    try {
      $this->poinzillaSetupCurlOptions($curl, $url, $data);

      // Esegue la richiesta
      $response = curl_exec($curl);
      $responseCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
      $error = curl_error($curl);

      // Log della risposta
      $this->poinzillaLogResponse($responseCode, $response, $error);

      return array(
        'response' => $response,
        'code' => $responseCode,
        'error' => $error
      );
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante la chiamata API: " . $e->getMessage(), 3);
      return array(
        'response' => null,
        'code' => 500,
        'error' => $e->getMessage()
      );
    } finally {
      curl_close($curl);
    }
  }

  /**
   * Configura le opzioni cURL per la richiesta
   *
   * @param resource $curl Handle cURL
   * @param string $url URL dell'endpoint
   * @param array $data Dati da inviare
   * @return void
   */
  private function poinzillaSetupCurlOptions($curl, $url, $data)
  {
    // Prepara il body della richiesta
    $jsonData = json_encode($data);
    if (json_last_error() !== JSON_ERROR_NONE) {
      throw new Exception("Errore nella codifica JSON: " . json_last_error_msg());
    }

    // Recupera la chiave pubblica per l'autenticazione
    $publicKey = Configuration::get('POINZILLA_PUBLIC_KEY');
    if (empty($publicKey)) {
      throw new Exception("Chiave privata non configurata");
    }

    // Configura le opzioni cURL
    curl_setopt_array($curl, array(
      CURLOPT_URL => $url,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_TIMEOUT => 30,
      CURLOPT_CUSTOMREQUEST => "POST",
      CURLOPT_POSTFIELDS => $jsonData,
      CURLOPT_HTTPHEADER => array(
        "Content-Type: application/json",
        self::AUTH_HEADER . ': ' . $publicKey
      ),
      CURLOPT_SSL_VERIFYPEER => true,  // Verifica il certificato SSL
      CURLOPT_SSL_VERIFYHOST => 2      // Verifica l'hostname
    ));
  }

  /**
   * Logga la risposta dell'API
   *
   * @param int $code Codice di risposta HTTP
   * @param string $response Corpo della risposta
   * @param string $error Eventuale errore cURL
   * @return void
   */
  private function poinzillaLogResponse($code, $response, $error)
  {
    if ($code == 200) {
      PoinzillaLogger::log("Chiamata API completata con successo", 1);
      PoinzillaLogger::log("Risposta: " . $response, 1);
    } else {
      if (!empty($error)) {
        PoinzillaLogger::log("Errore cURL: " . $error, 3);
      }
      PoinzillaLogger::log("Risposta API non valida. Codice: " . $code, 3);
      PoinzillaLogger::log("Dettagli risposta: " . $response, 3);
    }
  }
}