<?php
if (!defined('_PS_VERSION_')) {
  exit;
}

/**
 * Helper per gestire gli aggiornamenti del modulo Poinzilla.
 * Fornisce metodi statici per l'esecuzione degli script di upgrade, gestione delle configurazioni, hook e manipolazione del database.
 */
class PoinzillaUpgradeHelper
{
  /**
   * Esegue gli script di upgrade disponibili.
   * Controlla la versione corrente e applica gli upgrade necessari in ordine.
   * 
   * @param Module $module Istanza del modulo Poinzilla
   * @return bool True se tutti gli upgrade sono stati eseguiti con successo, False altrimenti
   */
  public static function poinzillaRunUpgradeScripts($module)
  {
    $currentVersion = Configuration::get('POINZILLA_VERSION');
    if (!$currentVersion) {
      PoinzillaLogger::log('Versione corrente non trovata', 3);
      return false;
    }

    $upgradeFiles = glob(_PS_MODULE_DIR_ . $module->name . '/upgrade/upgrade-*.php');
    if (!is_array($upgradeFiles)) {
      PoinzillaLogger::log('Nessun file di upgrade trovato', 2);
      return true;
    }

    usort($upgradeFiles, 'version_compare');

    foreach ($upgradeFiles as $file) {
      preg_match('/upgrade-(.*)\.php$/', $file, $matches);
      if (!isset($matches[1])) {
        PoinzillaLogger::log("File di upgrade non valido: {$file}", 3);
        continue;
      }

      $fileVersion = $matches[1];
      if (version_compare($currentVersion, $fileVersion, '<')) {
        require_once $file;
        $upgradeFunction = 'upgrade_module_' . str_replace('.', '_', $fileVersion);

        if (function_exists($upgradeFunction)) {
          PoinzillaLogger::log("Esecuzione script di upgrade: {$upgradeFunction}");
          if (!$upgradeFunction($module)) {
            PoinzillaLogger::log("Errore durante l'upgrade a {$fileVersion}", 3);
            return false;
          }
        } else {
          PoinzillaLogger::log("Funzione di upgrade non trovata: {$upgradeFunction}", 3);
          return false;
        }
      }
    }

    PoinzillaHelper::poinzillaClearModuleCache();
    Configuration::updateValue('POINZILLA_VERSION', $module->version);
    PoinzillaLogger::log('Aggiornamento Modulo Poinzilla completato');

    return true;
  }

  /**
   * Rimuove i file specificati come obsoleti nella cartella del modulo.
   *
   * @param array $files Lista di file obsoleti con percorso relativo alla cartella del modulo
   * @return void
   */
  public static function poinzillaRemoveObsoleteFiles(array $files)
  {
    $moduleDir = _PS_MODULE_DIR_ . 'poinzilla/';
    foreach ($files as $file) {
      $filePath = $moduleDir . $file;

      // Log del tentativo di rimozione del file
      if (class_exists('PoinzillaLogger')) {
        PoinzillaLogger::log("Tentativo di rimuovere il file: $filePath");
      }

      if (file_exists($filePath)) {
        unlink($filePath);
        if (class_exists('PoinzillaLogger')) {
          PoinzillaLogger::log("File obsoleto rimosso: $filePath");
        }
      } else {
        if (class_exists('PoinzillaLogger')) {
          PoinzillaLogger::log("File non trovato: $filePath");
        }
      }
    }
  }

  /**
   * Crea una colonna nel database se non esiste già.
   *
   * @param string $table Nome della tabella
   * @param string $column Nome della colonna da creare
   * @param string $definition Definizione SQL della colonna (es. "VARCHAR(255) NOT NULL")
   * @return bool True se la colonna è stata creata o esiste già, false se c'è stato un errore
   */
  public static function poinzillaCreateColumn($table, $column, $definition)
  {
    if (self::poinzillaColumnExists($table, $column)) {
      return true;
    }

    $sql = 'ALTER TABLE ' . _DB_PREFIX_ . pSQL($table) . ' ADD COLUMN `' . pSQL($column) . '` ' . pSQL($definition);
    if (!Db::getInstance()->execute($sql)) {
      PoinzillaLogger::log("Errore nella creazione della colonna $column nella tabella $table", 3);
      return false;
    }

    PoinzillaLogger::log("Colonna $column creata con successo nella tabella $table");
    return true;
  }

  /**
   * Controlla se una colonna esiste già in una tabella.
   *
   * @param string $table Nome della tabella
   * @param string $column Nome della colonna
   * @return bool True se la colonna esiste, false altrimenti
   */
  private static function poinzillaColumnExists($table, $column)
  {
    try {
      $result = Db::getInstance()->executeS('
                SHOW COLUMNS FROM `' . _DB_PREFIX_ . pSQL($table) . '` LIKE "' . pSQL($column) . '"'
      );

      return !empty($result);
    } catch (Exception $e) {
      PoinzillaLogger::log('Errore durante la verifica della colonna ' . $column . ': ' . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Registra una o più hook per il modulo.
   * 
   * @param array|string $hooks Nome della hook o array di nomi di hook da registrare
   * @param Module $module Istanza del modulo
   * @return bool True se tutte le hook sono state registrate con successo, false altrimenti
   */
  public static function poinzillaRegisterHook($hooks, $module)
  {
    if (!is_array($hooks)) {
      $hooks = [$hooks];
    }

    foreach ($hooks as $hook) {
      try {
        if (!$module->registerHook($hook)) {
          PoinzillaLogger::log("Errore durante la registrazione dell'hook {$hook}", 3);
          return false;
        }
        PoinzillaLogger::log("Hook {$hook} registrato con successo");
      } catch (Exception $e) {
        PoinzillaLogger::log("Errore durante la registrazione dell'hook {$hook}: " . $e->getMessage(), 3);
        return false;
      }
    }

    return true;
  }

  /**
   * Rimuove le hook specificate dal modulo.
   *
   * @param Module $module Istanza del modulo
   * @param array $hooks Lista di nomi di hook da rimuovere
   * @return bool True se tutte le hook sono state rimosse con successo, false altrimenti
   */
  public static function poinzillaUnregisterHooks($module, array $hooks)
  {
    foreach ($hooks as $hookName) {
      if ($module->isRegisteredInHook($hookName)) {
        if ($module->unregisterHook($hookName)) {
          PoinzillaLogger::log("Hook rimossa: {$hookName}");
        } else {
          PoinzillaLogger::log("Errore durante la rimozione della hook: {$hookName}", 3);
          return false;
        }
      }
    }
    return true;
  }

  /**
   * Aggiunge o aggiorna una configurazione di PrestaShop.
   * 
   * @param string $configName Nome della configurazione
   * @param mixed $defaultValue Valore di default della configurazione
   * @return bool True se la configurazione è stata aggiunta o aggiornata correttamente, false altrimenti
   */
  public static function poinzillaAddConfiguration($configName, $defaultValue)
  {
    if (Configuration::get($configName) === false) {
      if (!Configuration::updateValue($configName, $defaultValue)) {
        PoinzillaLogger::log("Errore durante l'aggiunta della configurazione $configName", 3);
        return false;
      }
      PoinzillaLogger::log("Configurazione $configName aggiunta con successo");
    } else {
      PoinzillaLogger::log("Configurazione $configName esiste già, nessuna modifica effettuata");
    }
    return true;
  }
}
