<?php
if (!defined('_PS_VERSION_')) {
  exit;
}

/**
 * Repository per la gestione dei clienti nel database
 * Gestisce tutte le operazioni CRUD sulla tabella poinzilla_customers
 */
class PoinzillaCustomerRepository
{
  /** @var string Nome della tabella nel database */
  private const TABLE_NAME = 'poinzilla_customers';

  /**
   * Mappa dei codici ISO delle lingue alle culture supportate da Poinzilla
   * @var array
   */
  private static $cultureMappings = [
    'it' => 'it',
    'de' => 'de',
    'fr' => 'fr',
    'es' => 'es',
    'en' => 'en',
    'gb' => 'en',
    'us' => 'en',
    'uk' => 'en'
    // Tutte le altre lingue verranno mappate su 'en' come fallback
  ];

  /**
   * Ottiene la cultura (lingua) del cliente nel formato richiesto da Poinzilla
   * 
   * @param Customer $customer Cliente di cui ottenere la cultura
   * @return string Codice cultura (it, de, fr, es, en)
   */
  public function poinzillaGetCustomerCultureId($customer)
  {
    // Ottiene l'oggetto Language dal cliente
    $langId = $customer->id_lang;
    
    // Debug delle informazioni del cliente
    PoinzillaLogger::log("Cliente #{$customer->id}: id_lang={$langId}", 1);
    
    // Se non è specificata, usa la lingua predefinita
    if (empty($langId)) {
      $langId = Configuration::get('PS_LANG_DEFAULT');
      PoinzillaLogger::log("Lingua cliente non trovata, uso lingua predefinita: " . $langId, 1);
    }
    
    // Ottiene l'oggetto Language completo
    $language = new Language($langId);
    
    if (!Validate::isLoadedObject($language)) {
      // Se l'oggetto Language non è valido, usa la lingua predefinita
      PoinzillaLogger::log("Oggetto lingua non valido, uso lingua predefinita", 2);
      $language = new Language(Configuration::get('PS_LANG_DEFAULT'));
    }
    
    $isoCode = strtolower($language->iso_code);
    
    PoinzillaLogger::log("ISO code della lingua del cliente: " . $isoCode, 1);
    
    // Gestione del codice ISO che potrebbe contenere il locale (es. it-it, en-us)
    $parts = explode('-', $isoCode);
    $baseIsoCode = $parts[0];
    
    // Prendi il codice cultura corrispondente o usa 'en' come default se non mappato
    $cultureId = isset(self::$cultureMappings[$baseIsoCode]) ? self::$cultureMappings[$baseIsoCode] : 'en';
    
    if (!isset(self::$cultureMappings[$baseIsoCode])) {
      PoinzillaLogger::log("ISO code '{$baseIsoCode}' non supportato da Poinzilla, uso 'en' come fallback", 1);
    }
    
    return $cultureId;
  }

  /**
   * Verifica se un cliente ha uno specifico stato di sincronizzazione
   * 
   * @param int $customerId ID del cliente
   * @param int $syncStatus Stato di sincronizzazione da verificare
   * @return bool True se il cliente ha lo stato specificato, False altrimenti
   */
  public function poinzillaIsCustomerSync($customerId, $syncStatus)
  {
    try {
      $db = Db::getInstance();
      $query = 'SELECT poinzilla_status 
                     FROM `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                     WHERE id_customer = ' . (int) $customerId;

      $result = $db->getValue($query);
      return $result == (int) $syncStatus;
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante la verifica dello stato di sincronizzazione: " . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Aggiorna lo stato di sincronizzazione di un cliente
   * Se il cliente non esiste, lo crea
   *
   * @param int $customerId ID del cliente
   * @param int $status Stato di sincronizzazione (0 = non sincronizzato, 1 = sincronizzato)
   * @return bool True se l'operazione è riuscita, False altrimenti
   */
  public function poinzillaUpdateCustomerSyncStatus($customerId, $status)
  {
    try {
      if ($this->poinzillaCustomerExists($customerId)) {
        return $this->poinzillaUpdateCustomer($customerId, $status);
      }
      return $this->poinzillaCreateCustomer($customerId, $status);
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante l'aggiornamento dello stato cliente: " . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Verifica se un cliente esiste nella tabella poinzilla_customers
   *
   * @param int $customerId ID del cliente
   * @return bool True se il cliente esiste, False altrimenti
   */
  private function poinzillaCustomerExists($customerId)
  {
    try {
      $db = Db::getInstance();
      $query = 'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                     WHERE id_customer = ' . (int) $customerId;
      return (bool) $db->getValue($query);
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante la verifica dell'esistenza del cliente: " . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Aggiorna lo stato di un cliente esistente
   *
   * @param int $customerId ID del cliente
   * @param int $status Nuovo stato
   * @return bool True se l'aggiornamento è riuscito, False altrimenti
   */
  private function poinzillaUpdateCustomer($customerId, $status)
  {
    try {
      $db = Db::getInstance();
      $updateQuery = 'UPDATE `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                          SET poinzilla_status = ' . (int) $status . ' 
                          WHERE id_customer = ' . (int) $customerId;

      if ($db->execute($updateQuery)) {
        PoinzillaLogger::log("Stato cliente aggiornato con successo. ID: $customerId, Stato: $status", 1);
        return true;
      }

      PoinzillaLogger::log("Errore durante l'aggiornamento dello stato cliente. ID: $customerId", 3);
      return false;
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante l'aggiornamento del cliente: " . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Crea un nuovo record per un cliente
   *
   * @param int $customerId ID del cliente
   * @param int $status Stato iniziale
   * @return bool True se la creazione è riuscita, False altrimenti
   */
  private function poinzillaCreateCustomer($customerId, $status)
  {
    try {
      PoinzillaLogger::log("Modulo Poinzilla ha inserito utente nella tabella: $customerId.", 1);

      $db = Db::getInstance();
      $insertQuery = 'INSERT INTO `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                          (id_customer, poinzilla_status) 
                          VALUES (' . (int) $customerId . ', ' . (int) $status . ')';

      if ($db->execute($insertQuery)) {
        PoinzillaLogger::log("Nuovo cliente creato con successo. ID: $customerId, Stato: $status", 1);
        return true;
      }

      PoinzillaLogger::log("Errore durante la creazione del cliente. ID: $customerId", 3);
      return false;
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante la creazione del cliente: " . $e->getMessage(), 3);
      return false;
    }
  }

  /**
   * Ottiene lo stato di sincronizzazione di un cliente
   *
   * @param int $customerId ID del cliente
   * @return int|null Stato di sincronizzazione o null se non trovato
   */
  public function poinzillaGetCustomerSyncStatus($customerId)
  {
    try {
      $db = Db::getInstance();
      $query = 'SELECT poinzilla_status 
                     FROM `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                     WHERE id_customer = ' . (int) $customerId;

      $result = $db->getValue($query);
      return $result !== false ? (int) $result : null;
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante il recupero dello stato cliente: " . $e->getMessage(), 3);
      return null;
    }
  }

  /**
   * Elimina un cliente dalla tabella poinzilla_customers
   *
   * @param int $customerId ID del cliente
   * @return bool True se l'eliminazione è riuscita, False altrimenti
   */
  public function poinzillaDeleteCustomer($customerId)
  {
    try {
      $db = Db::getInstance();
      $deleteQuery = 'DELETE FROM `' . _DB_PREFIX_ . self::TABLE_NAME . '` 
                          WHERE id_customer = ' . (int) $customerId;

      if ($db->execute($deleteQuery)) {
        PoinzillaLogger::log("Cliente eliminato con successo. ID: $customerId", 1);
        return true;
      }

      PoinzillaLogger::log("Errore durante l'eliminazione del cliente. ID: $customerId", 3);
      return false;
    } catch (Exception $e) {
      PoinzillaLogger::log("Errore durante l'eliminazione del cliente: " . $e->getMessage(), 3);
      return false;
    }
  }
}