<?php
if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Gestisce la sincronizzazione dei clienti con Poinzilla
 */
class PoinzillaCustomerSync
{
    /** @var Module */
    private $module;
    
    /** @var PoinzillaApiClient */
    private $apiClient;
    
    /** @var PoinzillaCustomerRepository */
    private $customerRepository;

    /**
     * @param Module $module Istanza del modulo
     */
    public function __construct($module)
    {
        $this->module = $module;
        $this->apiClient = new PoinzillaApiClient();
        $this->customerRepository = new PoinzillaCustomerRepository();
    }

    /**
     * Prepara e sincronizza i dati del cliente con Poinzilla
     *
     * @param Customer $customer Cliente da sincronizzare
     * @param bool $forceUpdate Se true, forza l'aggiornamento anche se il cliente è già sincronizzato
     * @return void
     */
    public function poinzillaPrepareCustomerData($customer, $forceUpdate = false)
    {
        // Verifica modalità setup
        if (PoinzillaHelper::poinzillaSetupMode(true)) {
            PoinzillaLogger::log("Poinzilla: Modalità di setup attiva, sincronizzazione saltata", 2);
            return;
        }

        // Verifica multistore
        if (!PoinzillaMultiStoreHelper::poinzillaCanSyncCustomer($customer->id)) {
            PoinzillaLogger::log("Poinzilla: Cliente non sincronizzabile - Modulo non installato nello shop di registrazione", 2);
            return;
        }

        // Verifica se il cliente è già stato sincronizzato, ma solo se non è un force update
        if (!$forceUpdate && PoinzillaHelper::isPoinzillaCustomerSync($customer->id, 1)) {
            PoinzillaLogger::log("Utente già sincronizzato", 1);
            return;
        }

        // DEBUG: Log del customer ID
        PoinzillaLogger::log("Customer ID che stiamo processando: " . $customer->id, 1);

        // Recupera le informazioni del gruppo
        $userGroups = [];
        if (isset($customer->id) && $customer->id) {
            // DEBUG: Log della query
            $query = 'SELECT id_group FROM '._DB_PREFIX_.'customer_group WHERE id_customer = '.(int)$customer->id;
            PoinzillaLogger::log("Query eseguita: " . $query, 1);

            $groups = Db::getInstance()->executeS($query);
            
            // DEBUG: Log dei gruppi trovati
            PoinzillaLogger::log("Gruppi trovati nel DB: " . json_encode($groups), 1);

            $userGroups = array_map(function($group) {
                return strval($group['id_group']);
            }, $groups);

            // DEBUG: Log dei gruppi dopo la trasformazione
            PoinzillaLogger::log("Gruppi dopo la trasformazione: " . json_encode($userGroups), 1);
        }

        // Ottiene la lingua del cliente e la mappa al formato richiesto
        $cultureId = $this->customerRepository->poinzillaGetCustomerCultureId($customer);
        PoinzillaLogger::log("CultureId determinata per il cliente: " . $cultureId, 1);

        $userData = array(
            'email' => $customer->email,
            'firstName' => $customer->firstname,
            'lastName' => $customer->lastname,
            'merchantCode' => Configuration::get('POINZILLA_MERCHANT_KEY'),
            'externalId' => (string) $customer->id,
            'birthDate' => null,
            'referralCode' => null,
            'group' => $userGroups,
            'cultureId' => $cultureId // Aggiunto campo cultureId
        );

        PoinzillaLogger::log("Payload della sincronizzazione: " . json_encode($userData, JSON_PRETTY_PRINT), 1);

        PoinzillaLogger::log("Poinzilla ottiene i dati utente", 1);
        return $this->poinzillaSyncCustomer($userData);
    }

    /**
     * Sincronizza i dati del cliente con l'API Poinzilla
     *
     * @param array $userData Dati del cliente da sincronizzare
     * @return void
     */
    private function poinzillaSyncCustomer($userData)
    {
        PoinzillaLogger::log("Poinzilla inizio sync HTTP", 1);
        $customerId = $this->module->context->customer->id;

        $response = $this->apiClient->poinzillaHTTPClient(
            POINZILLA_API_URL_V1 . "/api/External/Consumer", 
            $userData
        );
        
        $responseCode = $response['code'];
        $responseBody = $response['response'];

        if ($responseCode == 200) {
            PoinzillaLogger::log("Poinzilla ha sincronizzato l'utente con successo: $customerId.", 1);
            $this->customerRepository->poinzillaUpdateCustomerSyncStatus($customerId, 1);
        } elseif ($responseCode == 400) {
            $responseJson = json_decode($responseBody, true);
            if (isset($responseJson['detail']) && $responseJson['detail'] === "Consumatore già inserito") {
                PoinzillaLogger::log("Poinzilla: Consumatore già inserito, aggiornamento status per l'utente $customerId.", 1);
                $this->customerRepository->poinzillaUpdateCustomerSyncStatus($customerId, 1);
            } else {
                PoinzillaLogger::log("Errore: Poinzilla non è riuscito a sincronizzare l'utente $customerId. Codice risposta: $responseCode, Risposta: $responseBody", 3);
            }
        } else {
            PoinzillaLogger::log("Errore: Poinzilla non è riuscito a sincronizzare l'utente $customerId. Codice risposta: $responseCode, Risposta: $responseBody", 3);
        }
    }
}