<?php
if (!defined('_PS_VERSION_')) {
  exit;
}

/**
 * Controller per la gestione del frontend di Poinzilla
 * Gestisce la visualizzazione del widget e l'inizializzazione dell'SDK
 */
class PoinzillaWidgetController
{
  /** @var Module Istanza del modulo Poinzilla */
  private $module;

  /**
   * Costruttore
   * 
   * @param Module $module Istanza del modulo
   */
  public function __construct($module)
  {
    $this->module = $module;
  }

/**
   * Inizializza e restituisce l'HTML del widget Poinzilla
   * 
   * @return string HTML del widget o stringa vuota se non può essere mostrato
   */
  public function poinzillaWidgetInit()
  {
    $customerId = $this->module->context->customer->id ?: 'guest';

    // Log dei dati utente
    $currentUser = $this->module->context->customer;
    PoinzillaLogger::log('Dati utente corrente: ' . json_encode([
      'id' => $customerId,
      'nome' => $currentUser->firstname ?: 'ospite',
      'cognome' => $currentUser->lastname ?: 'ospite',
      'email' => $currentUser->email ?: 'ospite'
    ]));

    // Verifica modalità setup e permessi utente
    if (
      !PoinzillaHelper::poinzillaSetupModeAndUsersAllow(
        $this->module,
        "Poinzilla SDK: Non abilitato. Modalità setup attiva e utente corrente non autorizzato per anteprima. ID Cliente: " . $currentUser->email
      )
    ) {
      return '';
    }

    // Recupera dati referral dalla sessione
    $referralCode = $this->module->context->cookie->__get('poinzilla_referral_code');
    $defaultView = $this->module->context->cookie->__get('poinzilla_default_view');

    // Pulisce i dati dalla sessione
    $this->module->context->cookie->__unset('poinzilla_referral_code');
    $this->module->context->cookie->__unset('poinzilla_default_view');

    // Recupera i gruppi dell'utente
    $userGroups = [];
    if (isset($currentUser->id) && $currentUser->id) {
      $userGroups = $currentUser->getGroups();
    }

    // Gestione digest per utenti autenticati
    $isCustomerLoggedIn = isset($currentUser->id) && $currentUser->id;
    $userDigest = '';
    if ($isCustomerLoggedIn) {
      $userDigest = PoinzillaHelper::poinzillaGenerateDigest($currentUser->email);
    }

    // Genera HTML del widget
    return $this->poinzillaRenderWidgetHtml(
      $currentUser,
      $userDigest,
      $referralCode,
      $defaultView,
      $userGroups
    );
  }

  /**
   * Inizializza l'SDK di Poinzilla
   * 
   * @return string Script di inizializzazione o stringa vuota se non può essere caricato
   */
  public function poinzillaSDKInit()
  {
    if (
      !PoinzillaHelper::poinzillaSetupModeAndUsersAllow(
        $this->module,
        "Poinzilla SDK: Not enabled. Setup mode is active and current user is not allowed to preview. Customer ID: " . $this->module->context->customer->email
      )
    ) {
      return '';
    }

    return $this->poinzillaRenderSdkScript();
  }

/**
   * Genera l'HTML del widget Poinzilla
   * 
   * @param Customer $currentUser Utente corrente
   * @param string $userDigest Digest per l'autenticazione
   * @param string $referralCode Codice referral se presente
   * @param string $defaultView Vista di default
   * @param array $userGroups Gruppi dell'utente
   * @return string HTML del widget
   */
  private function poinzillaRenderWidgetHtml($currentUser, $userDigest, $referralCode, $defaultView, $userGroups)
  {
    // Convertiamo i gruppi in una stringa JSON
    $stringGroups = array_map(function($val) {
      return trim(strval($val)); // Trim any whitespace and convert to string
    }, $userGroups);
  
    $formattedGroups = json_encode($stringGroups);

    return '<div class="poinzilla-login-user-info" '
      . 'data-merchant-code="' . htmlspecialchars(Configuration::get('POINZILLA_MERCHANT_KEY')) . '" '
      . 'data-api-key="' . htmlspecialchars(Configuration::get('POINZILLA_PUBLIC_KEY')) . '" '
      . 'data-iframe-site-url="' . htmlspecialchars(POINZILLA_WIDGET_URL_V1) . '" '
      . 'data-api-url="' . htmlspecialchars(POINZILLA_API_URL_V1) . '" '
      . 'data-first-name="' . htmlspecialchars($currentUser->firstname ?: '') . '" '
      . 'data-last-name="' . htmlspecialchars($currentUser->lastname ?: '') . '" '
      . 'data-email="' . htmlspecialchars($currentUser->email ?: '') . '" '
      . 'data-digest="' . htmlspecialchars($userDigest) . '" '
      . 'data-referral-code="' . htmlspecialchars($referralCode) . '" '
      . 'data-default-view="' . htmlspecialchars($defaultView) . '" '
      . 'data-consumer-group=' . $formattedGroups . '>'
      . '</div>';
  }

  /**
   * Genera lo script di inizializzazione dell'SDK
   * 
   * @return string Script HTML per il caricamento dell'SDK
   */
  public function poinzillaRenderSdkScript()
  {
    $sdkUrl = POINZILLA_SDK_URL_V1;

    return <<<EOD
<script>
    (function () {
        var s = document.createElement('script');
        s.type = 'text/javascript';
        s.async = true;
        s.defer = true;
        s.src = '$sdkUrl';
        var x = document.getElementsByTagName('script')[0];
        x.parentNode.insertBefore(s, x);
    })();
</script>
EOD;
  }


}
