<?php

/**
 * Helper class per la gestione dei CartRule in Poinzilla.
 * Questa classe fornisce funzioni utili per operare sui CartRule (coupon) all'interno di PrestaShop,
 * come ottenere un CartRule specifico, recuperare l'ultimo CartRule aggiunto al carrello e rimuovere un CartRule.
 */
class PoinzillaCartRuleHelper
{
    /**
     * Ottiene un oggetto CartRule dal suo codice.
     * Questa funzione cerca il CartRule associato a un determinato codice, restituendo l'oggetto
     * CartRule se viene trovato e caricato correttamente. Se il CartRule non viene trovato o se 
     * il codice non è valido, viene restituito `false`.
     *
     * @param string $code Il codice del CartRule da cercare.
     * @return CartRule|false L'oggetto CartRule corrispondente al codice, o `false` se non trovato/valido.
     *
     * @example
     * ```php
     * $cartRule = PoinzillaCartRuleHelper::getCartRuleByCode('MYCOUPON123');
     * if ($cartRule) {
     *     // Procedi con l'oggetto CartRule
     * }
     * ```
     */
    public static function getCartRuleByCode($code)
    {
        // Ottiene l'ID del CartRule basato sul codice fornito
        $cartRuleId = CartRule::getIdByCode($code);
        if (!$cartRuleId) {
            return false; // Nessun CartRule trovato per il codice
        }

        // Crea l'oggetto CartRule con l'ID trovato
        $cartRule = new CartRule($cartRuleId);
        // Verifica se l'oggetto è stato caricato correttamente
        if (!Validate::isLoadedObject($cartRule)) {
            return false; // CartRule non valido o non caricato
        }

        return $cartRule; // Restituisce il CartRule caricato
    }

    /**
     * Ottiene l'ultimo CartRule aggiunto al carrello.
     * Questa funzione recupera l'ultimo CartRule associato al carrello passato come parametro, 
     * ordinando i CartRule in ordine decrescente e selezionando il più recente.
     * Se non ci sono CartRule, restituisce `false`.
     *
     * @param Cart $cart Il carrello dal quale ottenere l'ultimo CartRule.
     * @return CartRule|false L'oggetto CartRule più recente, o `false` se non ci sono CartRule.
     *
     * @example
     * ```php
     * $lastCartRule = PoinzillaCartRuleHelper::getLastCartRule($cart);
     * if ($lastCartRule) {
     *     // Esegui operazioni con l'ultimo CartRule aggiunto
     * }
     * ```
     */
    public static function getLastCartRule($cart)
    {
        // Query per ottenere l'ultimo CartRule associato al carrello
        $lastRule = Db::getInstance()->getRow('
            SELECT cr.* 
            FROM `'._DB_PREFIX_.'cart_cart_rule` ccr
            LEFT JOIN `'._DB_PREFIX_.'cart_rule` cr ON ccr.id_cart_rule = cr.id_cart_rule
            WHERE ccr.id_cart = '.(int)$cart->id.'
            ORDER BY ccr.id_cart_rule DESC'
        );

        // Restituisce `false` se nessun CartRule è stato trovato
        if (!$lastRule) {
            return false;
        }

        // Crea l'oggetto CartRule con l'ID trovato
        $cartRule = new CartRule($lastRule['id_cart_rule']);
        // Verifica se l'oggetto CartRule è stato caricato correttamente
        if (!Validate::isLoadedObject($cartRule)) {
            return false; // CartRule non valido o non caricato
        }

        return $cartRule; // Restituisce l'ultimo CartRule trovato
    }

    /**
     * Rimuove un CartRule specifico dal carrello.
     * Questa funzione elimina il CartRule associato al carrello e al CartRule specificato,
     * effettuando una query `DELETE` direttamente sul database.
     * Restituisce `true` se la rimozione ha successo, `false` altrimenti.
     *
     * @param Cart $cart Il carrello dal quale rimuovere il CartRule.
     * @param CartRule $cartRule Il CartRule da rimuovere.
     * @return bool `true` se il CartRule è stato rimosso con successo, `false` altrimenti.
     *
     * @example
     * ```php
     * if (PoinzillaCartRuleHelper::removeCartRule($cart, $cartRule)) {
     *     echo "CartRule rimosso con successo dal carrello.";
     * } else {
     *     echo "Errore nella rimozione del CartRule.";
     * }
     * ```
     */
    public static function removeCartRule($cart, $cartRule)
    {
        // Query di eliminazione per rimuovere la relazione tra il carrello e il CartRule
        return Db::getInstance()->execute('
            DELETE FROM `'._DB_PREFIX_.'cart_cart_rule` 
            WHERE id_cart = '.(int)$cart->id.' 
            AND id_cart_rule = '.(int)$cartRule->id
        );
    }
}
