<?php
if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Gestisce la sincronizzazione degli ordini con Poinzilla
 */
class PoinzillaOrderSync
{
    /** @var Module */
    private $module;
    
    /** @var PoinzillaApiClient */
    private $apiClient;

    /**
     * @param Module $module Istanza del modulo
     */
    public function __construct($module)
    {
        $this->module = $module;
        $this->apiClient = new PoinzillaApiClient();
    }

    /**
     * Aggiorna lo stato dell'ordine su Poinzilla
     *
     * @param array $params Parametri contenenti l'ID dell'ordine
     * @return void
     */
    public function poinzillaUpdateOrderStatus($params)
    {
        PoinzillaLogger::log("Poinzilla sta iniziando a sincronizzare l'ordine", 1);

        try {
            $order = new Order($params['id_order']);
            $customer = new Customer($order->id_customer);

            // Verifica modalità setup
            $setup_mode = PoinzillaHelper::poinzillaSetupMode(true);
            if ($setup_mode && !PoinzillaHelper::poinzillaOrderSyncConditions(
                $this->module, 
                $customer->id, 
                "Poinzilla sincronizzazione ordine skippata. Modalità di setup abilitata ed utente non abilitato ai test." . $customer->id
            )) {
                return;
            }

            // Verifica multistore
            if (!PoinzillaMultiStoreHelper::poinzillaCanSyncCustomer($customer->id)) {
                PoinzillaLogger::log("Poinzilla: Ordine non sincronizzabile - Modulo non installato nello shop di registrazione cliente", 2);
                return;
            }

            // Prepara i dati dell'ordine
            $payload = $this->poinzillaPrepareOrderData($order, $customer);
            
            // Invia i dati a Poinzilla
            $this->poinzillaSendOrderData($order, $payload);

        } catch (Exception $e) {
            PoinzillaLogger::log("Poinzilla Errore: " . $e->getMessage(), 3);
        }
    }

    /**
     * Prepara i dati dell'ordine per l'invio
     *
     * @param Order $order
     * @param Customer $customer
     * @return array
     */
    private function poinzillaPrepareOrderData($order, $customer)
    {
        // Prepara i prodotti
        $lineItems = $this->poinzillaPrepareProdotti($order);
        
        // Prepara i coupon
        $couponLines = $this->poinzillaPrepareCoupons($order);

        // Costruisce il payload
        return array(
            'id' => (int) $order->id,
            'status' => (string) $order->current_state,
            'customer_id' => (int) $customer->id,
            'consumer_email' => $customer->email,
            'total' => number_format((float)$order->total_products_wt, 2, '.', ''),
            'line_items' => $lineItems,
            'coupon_lines' => $couponLines
        );
    }

    /**
     * Prepara i dati dei prodotti dell'ordine
     *
     * @param Order $order
     * @return array
     */
    private function poinzillaPrepareProdotti($order)
    {
        $orderDetails = $order->getProducts();
        $lineItems = array();

        foreach ($orderDetails as $product) {
            $productCategories = Product::getProductCategoriesFull($product['product_id']);
            $allCategoryIds = array();

            foreach ($productCategories as $category) {
                $allCategoryIds[] = (string) $category['id_category'];

                // Include categorie padre
                $categoryObj = new Category($category['id_category']);
                $parentCategories = $categoryObj->getParentsCategories();
                foreach ($parentCategories as $parentCategory) {
                    $allCategoryIds[] = (string) $parentCategory['id_category'];
                }
            }

            // Rimuove categorie duplicate
            $uniqueCategoryIds = array_unique($allCategoryIds);

            $lineItems[] = array(
                'id' => $product['product_id'],
                'product_id' => $product['id_order_detail'],
                'productCat' => array_values($uniqueCategoryIds),
                'total' => (float) $product['total_price_tax_incl']
            );
        }

        return $lineItems;
    }

    /**
     * Prepara i dati dei coupon utilizzati nell'ordine
     *
     * @param Order $order
     * @return array
     */
    private function poinzillaPrepareCoupons($order)
    {
        $couponLines = array();
        $cartRules = $order->getCartRules();

        PoinzillaLogger::log("Numero di coupon trovati: " . count($cartRules), 1);
        
        if (empty($cartRules)) {
            PoinzillaLogger::log("Attenzione: nessun coupon trovato per l'ordine", 2);
            return $couponLines;
        }

        foreach ($cartRules as $cartRule) {
            PoinzillaLogger::log("Dati del coupon: " . json_encode($cartRule), 1);

            // Controllo dei valori richiesti
            if (!isset($cartRule['id_cart_rule']) || !isset($cartRule['value'])) {
                PoinzillaLogger::log("Errore: dati del coupon mancanti: " . json_encode($cartRule), 3);
                continue;
            }

            // Ottengo le info dall'oggetto CartRule
            $cartRuleObj = new CartRule($cartRule['id_cart_rule']);
            if (Validate::isLoadedObject($cartRuleObj)) {
                $discountValue = round((float) $cartRule['value'], 2);
                $couponLines[] = array(
                    'id' => (string) $cartRule['id_cart_rule'],
                    'code' => (string) $cartRuleObj->code,
                    'discount' => $discountValue,
                );
            } else {
                PoinzillaLogger::log("Errore: impossibile caricare CartRule con id: " . $cartRule['id_cart_rule'], 3);
            }
        }

        return $couponLines;
    }

    /**
     * Invia i dati dell'ordine a Poinzilla
     *
     * @param Order $order
     * @param array $payload
     * @return void
     */
    private function poinzillaSendOrderData($order, $payload)
    {
        $payloadJson = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        PoinzillaLogger::log("Poinzilla ha preparato il payload dell'ordine", 1);
        PoinzillaLogger::log("Poinzilla: " . $payloadJson, 1);

        $response = $this->apiClient->poinzillaHTTPClient(
          POINZILLA_API_URL_V1 . "/api/External/Order",
          $payload
        );

        $responseCode = $response['code'];
        $responseBody = $response['response'];

        PoinzillaLogger::log("Poinzilla: Risposta HTTP Ricevuta", 1);

        if ($responseCode == 200) {
            PoinzillaLogger::log("Poinzilla ha sincronizzato l'ordine id: " . $order->id, 1);
        } else {
            PoinzillaLogger::log("Errore: Poinzilla payload: " . $payloadJson, 3);
            PoinzillaLogger::log("Errore: Poinzilla non è riuscito a sincronizzare l'ordine. Codice risposta: $responseCode, Risposta: $responseBody", 3);
        }
    }
}