<?php

/**
 * Controller per la gestione delle regole carrello nel back office.
 * Gestisce l'inizializzazione degli assets e la logica specifica delle cart rules
 * quando la funzionalità extra è abilitata.
 */
class PoinzillaCartRulesBackOfficeController
{
    /** @var Module */
    private $module;

    public function __construct($module)
    {
        $this->module = $module;
    }

    /**
     * Inizializza gli assets necessari per la pagina delle regole carrello.
     * Se non siamo nella pagina corretta o la feature non è attiva, 
     * termina senza fare nulla.
     */
    public function poinzillaInitCartRulesAssets()
    {
        if (!$this->poinzillaIsCartRulesPage()) {
          return;
        }

        // Utilizziamo l'helper per verificare se la funzionalità è attiva
        if (!PoinzillaHelper::poinzillaIsExtraCartRulesFieldsEnabled()) {
            return;
        }

        $this->poinzillaAddExclusiveCouponJs();
        $this->poinzillaAddCartRulesScript();
    }

    /**
     * Verifica se ci troviamo nella pagina delle regole carrello.
     * 
     * @return bool
     */
    private function poinzillaIsCartRulesPage()
    {
        return $this->module->context->controller instanceof AdminCartRulesController;
    }

    /**
     * Aggiunge la variabile JavaScript per i coupon esclusivi.
     * Questa variabile viene utilizzata nel form delle regole carrello
     * per gestire la logica dei coupon esclusivi.
     */
    private function poinzillaAddExclusiveCouponJs()
    {
        $exclusive_value = $this->poinzillaGetExclusiveCouponValue();
        
        Media::addJsDef([
            'pz_exclusive_coupon' => $exclusive_value
        ]);
    }

    /**
     * Recupera il valore del coupon esclusivo dal database.
     * Se l'ID della cart rule non è presente, restituisce 0.
     * 
     * @return int
     */
    private function poinzillaGetExclusiveCouponValue()
    {
        $exclusive_value = 0;
        
        if ($id_cart_rule = Tools::getValue('id_cart_rule')) {
            $sql = 'SELECT pz_exclusive_coupon FROM ' . _DB_PREFIX_ . 'cart_rule WHERE id_cart_rule = ' . (int) $id_cart_rule;
            $exclusive_value = (int) Db::getInstance()->getValue($sql);
            
            $this->poinzillaLogExclusiveValue($exclusive_value);
        }

        return $exclusive_value;
    }

    /**
     * Aggiunge lo script JavaScript per la gestione delle regole carrello.
     * Questo script gestisce la logica custom del form delle regole carrello.
     */
    private function poinzillaAddCartRulesScript()
    {
        $this->module->context->controller->addJS(
            $this->module->getPathUri() . 'views/js/poinzilla-cart-rules-form.js'
        );
    }

    /**
     * Registra nel log il valore del coupon esclusivo.
     * Utile per il debugging e il monitoraggio.
     * 
     * @param int $value
     */
    private function poinzillaLogExclusiveValue($value)
    {
        PoinzillaLogger::log("Valore exclusive_coupon recuperato: " . $value);
    }
}