<?php

class PoinzillaHelper
{
  public static function autoloadHelper()
  {
    spl_autoload_register(function ($class) {
      $class_file = __DIR__ . '/../helpers/' . $class . '.php';
      if (file_exists($class_file)) {
        require_once $class_file;
      }
    });
  }

  public static function poinzillaSetupMode($checkOn)
  {

    $setup_mode_on = (bool) Configuration::get('POINZILLA_SETUP_MODE');
    return $checkOn === $setup_mode_on;
  }

  public static function poinzillaIsAllowedCustomer($customer)
  {

    $allowed_customers = array_map('trim', explode(',', Configuration::get('POINZILLA_ALLOWED_CUSTOMERS')));

    return in_array($customer->email, $allowed_customers);
  }

  public static function poinzillaSetupModeAndUsersAllow($module, $logMessage = '')
  {
    $setup_mode = self::poinzillaSetupMode(true);
    $current_user_allowed = self::poinzillaIsAllowedCustomer($module->context->customer);

    if ($setup_mode && !$current_user_allowed) {
      if ($logMessage) {
        PoinzillaLogger::log($logMessage, 2);
      }
      return false;
    }

    return true;
  }

  public static function poinzillaOrderSyncConditions($module, $customerId, $logMessage = '')
  {
    $setup_mode = self::poinzillaSetupMode(true);

    if ($setup_mode) {
      $customer = new Customer($customerId);
      $current_user_allowed = self::poinzillaIsAllowedCustomer($customer);

      if (!$current_user_allowed) {
        if ($logMessage) {
          PoinzillaLogger::log($logMessage, 2);
        }
        return false;
      }
    }

    return true;
  }

  public static function poinzillaGenerateDigest($userEmail)
  {
    $merchant_key = Configuration::get('POINZILLA_MERCHANT_KEY');
    $private_key = Configuration::get('POINZILLA_PRIVATE_KEY');

    return hash_hmac('sha256', $merchant_key . $userEmail, $private_key);
  }

  public static function isPoinzillaCustomerSync($customerId, $syncStatus)
  {
    $repository = new PoinzillaCustomerRepository();
    return $repository->poinzillaIsCustomerSync($customerId, $syncStatus);
  }

  public static function poinzillaClearModuleCache()
  {
    $moduleTemplates = [
      'module:poinzilla/views/templates/hook/frontend.tpl',
    ];


    $context = Context::getContext();

    foreach ($moduleTemplates as $template) {
      $context->smarty->clearCache($template);
      PoinzillaLogger::log('Poinzilla Cache cancellata per il modulo: ' . $template);
    }

    Cache::clean('poinzilla');
    Tools::clearSmartyCache();
    Tools::clearCompile();
    Tools::clearCache();

    Hook::exec('actionModuleRegisterHookAfter', ['module_name' => 'poinzilla']);

    PoinzillaLogger::log('Poinzilla cancellazione cache completata dopo salvataggio settings o aggiornamento');
  }

  /**
   * Verifica se il campo di configurazione POINZILLA_EXTRA_CART_RULES_FIELDS_ENABLED è abilitato.
   *
   * Questa funzione restituisce un valore booleano che indica se il campo di configurazione
   * `POINZILLA_EXTRA_CART_RULES_FIELDS_ENABLED` è impostato su `true`. 
   * Può essere utilizzato per attivare logiche aggiuntive quando il campo è abilitato.
   *
   * @return bool `true` se il campo è abilitato, `false` altrimenti.
   */
  public static function poinzillaIsExtraCartRulesFieldsEnabled()
  {
    return (bool) Configuration::get('POINZILLA_EXTRA_CART_RULES_FIELDS_ENABLED');
  }

}