<?php

if (!defined('_PS_VERSION_')) {
  exit;
}

/**
 * Costanti di configurazione per gli endpoint Poinzilla
 * Utilizzate per le chiamate API e il caricamento di widget/SDK
 */
define('POINZILLA_API_URL_V1', 'https://api.poinzilla.com');
define('POINZILLA_WIDGET_URL_V1', 'https://widget.poinzilla.com');
define('POINZILLA_SDK_URL_V1', "https://sdk.poinzilla.com/sdk.umd.js");


/**
 * Inclusione dei file core necessari per il funzionamento base del modulo
 * 
 * Questi file sono richiesti fin dall'inizio perché:
 * - PoinzillaHelper: contiene funzioni di utilità usate in tutto il modulo
 * - PoinzillaUpgradeHelper: gestisce gli aggiornamenti del modulo
 * - PoinzillaLogger: gestisce il logging delle operazioni
 * - PoinzillaCartRuleForm: gestisce i form delle regole carrello
 * - PoinzillaCartRule: gestisce le regole carrello
 */
include_once __DIR__ . '/classes/Core/PoinzillaLogger.php';
include_once __DIR__ . '/classes/Api/PoinzillaApiClient.php';
include_once __DIR__ . '/classes/Service/PoinzillaCustomerSync.php';
include_once __DIR__ . '/classes/Service/PoinzillaOrderSync.php';
include_once __DIR__ . '/classes/Repository/PoinzillaCustomerRepository.php';
include_once __DIR__ . '/classes/Model/PoinzillaCartRule.php';
include_once __DIR__ . '/classes/Model/PoinzillaCartRuleForm.php';
include_once __DIR__ . '/classes/Validation/PoinzillaCartRuleValidator.php';
include_once __DIR__ . '/helpers/PoinzillaHelper.php';
include_once __DIR__ . '/helpers/PoinzillaUpgradeHelper.php';
include_once __DIR__ . '/helpers/PoinzillaCartRuleHelper.php';
include_once __DIR__ . '/helpers/PoinzillaMultiStoreHelper.php';
include_once __DIR__ . '/handlers/PoinzillaCartRuleHandler.php';
include_once __DIR__ . '/controllers/admin/PoinzillaAdminController.php';
require_once __DIR__ . '/controllers/admin/PoinzillaCartRulesBackOfficeController.php';
include_once __DIR__ . '/controllers/front/PoinzillaWidgetController.php';
include_once __DIR__ . '/controllers/front/PoinzillaReferralController.php';
include_once __DIR__ . '/controllers/front/PoinzillaCartRuleController.php';
class Poinzilla extends Module
{
  public $identifier = 'id_poinzilla';
  protected $config_form = false;
  public $context;
  public function __construct()
  {
    $this->name = 'poinzilla';
    $this->tab = 'advertising_marketing';
    $this->version = '1.2.5';
    $this->author = 'Feedaty Srl';
    $this->need_instance = 0;
    $this->bootstrap = true;

    parent::__construct();

    $this->displayName = $this->l('Poinzilla');
    $this->description = $this->l('Connettore ufficiale di Poinzilla. Lancia il tuo programma di loyalty personalizzato e incrementa il valore dei tuoi clienti.');

    $this->confirmUninstall = $this->l('Sei sicuro di procedere?');

    $this->ps_versions_compliancy = array('min' => '1.6', 'max' => _PS_VERSION_);

    $this->cartRuleForm = new PoinzillaCartRuleForm();
    $this->widgetController = new PoinzillaWidgetController($this);
    $this->cartRuleController = new PoinzillaCartRuleControllerModuleFrontController($this);
  }

  public function install()
  {

    $currentVersion = Configuration::get('POINZILLA_VERSION');

    if (!$currentVersion) {
      Configuration::updateValue('POINZILLA_VERSION', $this->version);
    } elseif ($currentVersion !== $this->version) {
      $this->upgrade();
    }

    Configuration::updateValue('POINZILLA_MERCHANT_KEY', '');
    Configuration::updateValue('POINZILLA_PUBLIC_KEY', '');
    Configuration::updateValue('POINZILLA_PRIVATE_KEY', '');
    Configuration::updateValue('POINZILLA_SETUP_MODE', 1);
    Configuration::updateValue('POINZILLA_ALLOWED_CUSTOMER', '');
    Configuration::updateValue('POINZILLA_ENABLE_LOG', 0);
    Configuration::updateValue('POINZILLA_EXTRA_CART_RULES_FIELDS_ENABLED', 0);
    Configuration::updateValue('POINZILLA_VERSION', $this->version);

    require_once _PS_MODULE_DIR_ . 'poinzilla/sql/install.php';

    return parent::install()
      && $this->registerHook('actionOrderStatusPostUpdate')
      && $this->registerHook('displayBackOfficeHeader')
      && $this->registerHook('displayAfterBodyOpeningTag')
      && $this->registerHook('actionFrontControllerInitBefore')
      && $this->registerHook('displayHeader')
      && $this->registerHook('actionAuthentication')
      && $this->registerHook('actionCustomerAccountAdd')
      && $this->registerHook('actionObjectCartRuleAddBefore')
      && $this->registerHook('actionObjectCartRuleAddAfter')
      && $this->registerHook('actionObjectCartRuleUpdateBefore')
      && $this->registerHook('actionCartSave')
      && $this->registerHook('actionAjaxDieBefore')
      && $this->registerHook('actionFrontControllerInitBefore')
      && $this->registerHook('actionCustomerAccountUpdate')
      && $this->registerHook('actionObjectCustomerUpdateAfter');
  }

  public function uninstall()
  {
    Configuration::deleteByName('POINZILLA_MERCHANT_KEY', '');
    Configuration::deleteByName('POINZILLA_PUBLIC_KEY', '');
    Configuration::deleteByName('POINZILLA_PRIVATE_KEY', '');
    Configuration::deleteByName('POINZILLA_SETUP_MODE', '');
    Configuration::deleteByName('POINZILLA_ALLOWED_CUSTOMER', '');
    Configuration::deleteByName('POINZILLA_ENABLE_LOG', '');
    Configuration::deleteByName('POINZILLA_EXTRA_CART_RULES_FIELDS_ENABLED', '');
    Configuration::deleteByName('POINZILLA_VERSION', '');

    require_once _PS_MODULE_DIR_ . 'poinzilla/sql/uninstall.php';

    return parent::uninstall()
      && $this->registerHook('actionOrderStatusPostUpdate')
      && $this->registerHook('displayBackOfficeHeader')
      && $this->registerHook('displayAfterBodyOpeningTag')
      && $this->registerHook('actionFrontControllerInitBefore')
      && $this->registerHook('displayHeader')
      && $this->registerHook('actionAuthentication')
      && $this->registerHook('actionCustomerAccountAdd')
      && $this->registerHook('actionObjectCartRuleAddBefore')
      && $this->registerHook('actionObjectCartRuleAddAfter')
      && $this->registerHook('actionObjectCartRuleUpdateBefore')
      && $this->registerHook('actionCartSave')
      && $this->registerHook('actionAjaxDieBefore')
      && $this->registerHook('actionFrontControllerInitBefore')
      && $this->registerHook('actionCustomerAccountUpdate')
      && $this->registerHook('actionObjectCustomerUpdateAfter');
  }

  /**
   * Esegue gli script di aggiornamento del modulo
   * Questo metodo viene chiamato quando viene rilevata una nuova versione del modulo
   * 
   * @param string|null $version La versione target dell'aggiornamento
   * @return bool True se l'aggiornamento è avvenuto con successo, False altrimenti
   */
  public function upgrade($version = null)
  {
    return PoinzillaUpgradeHelper::poinzillaRunUpgradeScripts($this);
  }

  /**
   * Restituisce il contenuto della pagina di configurazione del modulo nel back office
   * Questo metodo viene chiamato quando si accede alla configurazione del modulo da PrestaShop
   * 
   * @return string HTML della pagina di configurazione
   */

  public function getContent()
  {
    $adminController = new PoinzillaAdminController($this);
    return $adminController->getContent();
  }

  /**
   * Hook visualizzato nell'header
   * Inizializza l'SDK Poinzilla
   * 
   * @return string Script di inizializzazione SDK
   */
  public function hookDisplayHeader()
  {
    return $this->widgetController->poinzillaSDKInit();
  }

  /**
   * Hook visualizzato dopo il tag body
   * Inizializza il widget Poinzilla
   * 
   * @return string HTML del widget
   */
  public function hookDisplayAfterBodyOpeningTag()
  {
    return $this->widgetController->poinzillaWidgetInit();
  }

  /**
   * Hook chiamato dopo l'aggiornamento dello stato di un ordine
   * Gestisce la sincronizzazione dello stato dell'ordine con Poinzilla
   * @return mixed Risultato dell'operazione di aggiornamento stato su Poinzilla
   */
  public function hookActionOrderStatusPostUpdate($params)
  {
    $orderSync = new PoinzillaOrderSync($this);
    return $orderSync->poinzillaUpdateOrderStatus($params);
  }

  /**
   * Hook chiamato dopo l'autenticazione di un cliente
   * Gestisce la sincronizzazione dei dati cliente con Poinzilla dopo il login
   * 
   * @param array $params Parametri dell'hook contenenti:
   *                     - customer: oggetto Customer con i dati del cliente autenticato
   * @return void|mixed Risultato dell'operazione di sincronizzazione
   */
  public function hookActionAuthentication($params)
  {
    PoinzillaLogger::log("Poinzilla hookActionAuthentication: Login effettuato. Inizio a preparare i dati del customer", 1);

    $customerSync = new PoinzillaCustomerSync($this);
    return $customerSync->poinzillaPrepareCustomerData($params['customer']);
  }

  /**
   * Hook chiamato dopo la creazione di un nuovo account cliente
   * Sincronizza i dati del nuovo cliente con Poinzilla
   * 
   * @param array $params Parametri dell'hook contenente il nuovo cliente
   * @return mixed Risultato della sincronizzazione
   */
  public function hookActionCustomerAccountAdd($params)
  {
    PoinzillaLogger::log("Poinzilla: Nuovo account cliente creato", 1);

    $customerSync = new PoinzillaCustomerSync($this);
    return $customerSync->poinzillaPrepareCustomerData($params['newCustomer']);
  }

  /**
   * Hook chiamato prima dell'inizializzazione del controller front
   * Gestisce i redirect dei referral
   */
  public function hookActionFrontControllerInitBefore()
  {
    $referralController = new PoinzillaReferralController($this);
    $referralController->poinzillaHandleReferralRedirect();
  }

  /**
   * Hook visualizzata nell'header del back office.
   * Gestisce l'inizializzazione degli assets per le regole carrello quando:
   * - Ci troviamo nella pagina delle regole carrello (AdminCartRulesController)
   * - La funzionalità extra delle regole carrello è abilitata
   * 
   * Questa hook viene utilizzata per:
   * - Caricare gli script JS necessari per il form delle regole carrello
   * - Inizializzare le variabili JS per i coupon esclusivi
   * 
   * @return void|string Output della hook (se presente)
   */
  public function hookDisplayBackOfficeHeader()
  {
    $controller = new PoinzillaCartRulesBackOfficeController($this);
    return $controller->poinzillaInitCartRulesAssets();
  }

  /**
   * Hook chiamato prima della creazione di una nuova cart rule
   * Salva temporaneamente il valore del campo exclusive_coupon per il successivo salvataggio
   * 
   * @param array $params Parametri dell'hook contenenti l'oggetto CartRule
   * @return mixed Risultato dell'operazione di salvataggio temporaneo
   */
  public function hookActionObjectCartRuleAddBefore($params)
  {
    PoinzillaLogger::log("=== INIZIO hookActionObjectCartRuleAddBefore ===", 1);
    return $this->cartRuleForm->poinzillaSaveTemporaryExclusiveCouponValue($params);
  }

  /**
   * Hook chiamato dopo la creazione di una nuova cart rule
   * Salva definitivamente il valore di exclusive_coupon nel database
   * 
   * @param array $params Parametri dell'hook contenenti l'oggetto CartRule con ID assegnato
   * @return mixed Risultato dell'operazione di salvataggio definitivo
   */
  public function hookActionObjectCartRuleAddAfter($params)
  {
    PoinzillaLogger::log("=== INIZIO hookActionObjectCartRuleAddAfter ===", 1);
    return $this->cartRuleForm->poinzillaPersistExclusiveCouponValue($params);
  }

  /**
   * Hook chiamato prima dell'aggiornamento di una cart rule esistente
   * Gestisce l'aggiornamento del campo exclusive_coupon da interfaccia amministrativa
   * 
   * @param array $params Parametri dell'hook contenenti l'oggetto CartRule da aggiornare
   * @return mixed Risultato dell'operazione di aggiornamento
   */
  public function hookActionObjectCartRuleUpdateBefore($params)
  {
    PoinzillaLogger::log("=== INIZIO hookActionObjectCartRuleUpdateBefore ===", 1);
    return $this->cartRuleForm->poinzillaUpdateExclusiveCouponValue($params);
  }

  /**
   * Hook chiamato prima della risposta AJAX.
   * Questo metodo intercetta le richieste AJAX per validare l'aggiunta di un coupon
   * al carrello prima che l'operazione venga completata. In caso di validazione fallita,
   * restituisce una risposta JSON con un errore.
   *
   * @param array $params Parametri forniti dal hook.
   */
  public function hookActionAjaxDieBefore($params)
  {
    PoinzillaLogger::log("Hook called", 1);
    if (!PoinzillaHelper::poinzillaIsExtraCartRulesFieldsEnabled()) {
      return;
    }
    if (
      !in_array(Tools::getValue('action'), ['update', 'AddVoucher', 'addVoucher']) ||
      Tools::getValue('addDiscount') != 1 ||
      !Tools::getValue('discount_name')
    ) {
      return;
    }

    PoinzillaLogger::log("=== INIZIO VALIDAZIONE AJAX COUPON ===", 1);

    try {
      $this->cartRuleController->poinzillaAddCartRuleToCart(
        Context::getContext()->cart,
        Tools::getValue('discount_name')
      );
      PoinzillaLogger::log("Validazione AJAX completata con successo", 1);
    } catch (PrestaShopException $e) {
      PoinzillaLogger::log("Validazione AJAX fallita: " . $e->getMessage(), 3);
      $this->ajaxDie([
        'errors' => [$e->getMessage()],
        'hasError' => true
      ]);
    }
  }


  /**
   * Hook chiamato durante il salvataggio del carrello.
   * Questo metodo gestisce la validazione e l'aggiunta di un coupon al carrello
   * durante il salvataggio del carrello. In caso di fallimento della validazione,
   * logga un errore e interrompe l'operazione.
   *
   * @param array $params Parametri forniti dal hook, inclusivo del carrello.
   */
  public function hookActionCartSave($params)
  {
    if (!PoinzillaHelper::poinzillaIsExtraCartRulesFieldsEnabled()) {
      return;
    }

    if (!isset($params['cart'])) {
      return;
    }

    $couponCode = Tools::getValue('coupon_code');
    if (empty($couponCode)) {
      return;
    }

    try {
      $this->cartRuleController->poinzillaAddCartRuleToCart(Context::getContext()->cart, $couponCode);
    } catch (PrestaShopException $e) {
      PoinzillaLogger::log("Errore durante l'aggiunta del coupon: " . $e->getMessage(), 3);
    }
  }

  /**
   * Metodo helper per terminare una richiesta AJAX con una risposta JSON.
   * Questo metodo imposta l'intestazione della risposta a JSON e termina l'esecuzione
   * restituendo i dati specificati nella risposta.
   *
   * @param array $response Dati di risposta da restituire alla richiesta AJAX.
   */
  private function ajaxDie($response)
  {
    header('Content-Type: application/json');
    die(json_encode($response));
  }

  /**
   * Hook chiamato dopo l'aggiornamento dell'account cliente dal front-office
   */
  public function hookActionCustomerAccountUpdate($params)
  {
    PoinzillaLogger::log("Poinzilla: Aggiornamento account cliente dal front-office", 1);

    $customerSync = new PoinzillaCustomerSync($this);
    return $customerSync->poinzillaPrepareCustomerData($params['customer'], true);
  }

  /**
   * Hook chiamato dopo l'aggiornamento del cliente dal back-office
   */
  public function hookActionObjectCustomerUpdateAfter($params)
  {
    PoinzillaLogger::log("Poinzilla: Aggiornamento cliente dal back-office", 1);

    $customer = new Customer($params['object']->id);
    
    $customerSync = new PoinzillaCustomerSync($this);
    return $customerSync->poinzillaPrepareCustomerData($customer, true);
  }

}
