<?php
defined('ABSPATH') or exit;

/**
 * Gestione dell'invio degli ordini a Poinzilla
 *
 * @link       https://www.zoorate.com
 * @since      1.0.0
 *
 * @package    Poinzilla
 * @subpackage Poinzilla/admin
 */

/**
 * Classe per la gestione dell'invio degli ordini a Poinzilla
 *
 * Questa classe gestisce sia l'invio singolo che massivo degli ordini a Poinzilla
 * tramite l'interfaccia di amministrazione di WooCommerce.
 *
 * @since      1.0.0
 * @package    Poinzilla
 * @subpackage Poinzilla/admin
 * @author     Zoorate srl <info@zoorate.com>
 */
class Poinzilla_PushOrder
{


  /**
   * Inizializza la classe e registra i vari hook necessari
   *
   * @since    1.0.0
   */
  public function __construct()
  {
    // Carica le dipendenze necessarie
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-logger.php';
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-order-sync.php';
    
    // Registra le bulk actions per gli ordini (manteniamo anche questo metodo come alternativa)
    add_filter('bulk_actions-edit-shop_order', array($this, 'poinzilla_register_bulk_actions'));
    add_filter('bulk_actions-woocommerce_page_wc-orders', array($this, 'poinzilla_register_bulk_actions'));
    
    // Gestisce l'esecuzione delle bulk actions
    add_filter('handle_bulk_actions-edit-shop_order', array($this, 'poinzilla_handle_bulk_action'), 10, 3);
    add_filter('handle_bulk_actions-woocommerce_page_wc-orders', array($this, 'poinzilla_handle_bulk_action'), 10, 3);
    
    // Mostra i messaggi di notifica dopo l'esecuzione delle bulk actions
    add_action('admin_notices', array($this, 'poinzilla_bulk_action_admin_notice'));
    
    // Aggiungi il pulsante personalizzato sopra la tabella degli ordini
    add_action('woocommerce_admin_order_bulk_actions', array($this, 'poinzilla_add_custom_bulk_button'), 10);
    add_action('manage_posts_extra_tablenav', array($this, 'poinzilla_add_custom_bulk_button_legacy'), 10, 1);
    
    // Aggiungi il modale di progresso per l'interfaccia HPS
    add_action('admin_footer', array($this, 'poinzilla_add_progress_modal_hps'));
    
    // Registra lo script AJAX per processare gli ordini selezionati
    add_action('wp_ajax_poinzilla_bulk_process_orders', array($this, 'poinzilla_bulk_process_orders_ajax'));
    
    // Aggiungi gli script e gli stili necessari
    add_action('admin_enqueue_scripts', array($this, 'poinzilla_enqueue_bulk_scripts'));
  }

  /**
   * Registra l'azione massiva nel dropdown delle azioni
   */
  public function poinzilla_register_bulk_actions($bulk_actions)
  {
    $bulk_actions['poinzilla_sync_orders'] = __('Invia a Poinzilla', 'poinzilla');
    return $bulk_actions;
  }

  /**
   * Gestisce l'esecuzione dell'azione massiva
   * Questa funzione è necessaria per registrare l'azione di gruppo,
   * ma l'elaborazione effettiva viene gestita via JavaScript
   */
  public function poinzilla_handle_bulk_action($redirect_to, $action, $post_ids)
  {
    // Non facciamo nulla qui, poiché l'azione viene intercettata via JavaScript
    // e non vogliamo che la pagina venga ricaricata
    return $redirect_to;
  }

  /**
   * Mostra il messaggio di notifica dopo l'esecuzione dell'azione massiva
   */
  public function poinzilla_bulk_action_admin_notice()
  {
    if (empty($_REQUEST['poinzilla_bulk_sync'])) {
      return;
    }

    $processed = isset($_REQUEST['processed']) ? intval($_REQUEST['processed']) : 0;
    $failed = isset($_REQUEST['failed']) ? intval($_REQUEST['failed']) : 0;
    $total = isset($_REQUEST['total']) ? intval($_REQUEST['total']) : 0;

    $class = ($failed === 0) ? 'notice-success' : 'notice-warning';
    $message = sprintf(
      __('Poinzilla: %d ordini su %d sono stati sincronizzati con successo.', 'poinzilla'),
      $processed,
      $total
    );

    if ($failed > 0) {
      $message .= ' ' . sprintf(
        __('%d ordini non sono stati sincronizzati. Controlla i log per maggiori dettagli.', 'poinzilla'),
        $failed
      );
    }

    printf('<div class="notice %s poinzilla-bulk-notice is-dismissible"><p>%s</p></div>', esc_attr($class), esc_html($message));
  }

  /**
   * Registra la metabox per l'invio manuale degli ordini
   */
  public function poinzilla_register_meta_box()
  {
    add_meta_box('poinzilla-order-meta-box', esc_html__('Poinzilla - Invio manuale', 'poinzilla'), array($this, 'poinzilla_order_meta_box_callback'), 'shop_order', 'side', 'high');
    add_meta_box('poinzilla-order-meta-box', esc_html__('Poinzilla - Invio manuale', 'poinzilla'), array($this, 'poinzilla_order_meta_box_callback'), 'woocommerce_page_wc-orders', 'side', 'high');
  }

  /**
   * Callback per la metabox di invio manuale
   */
  public function poinzilla_order_meta_box_callback($meta_id)
  {
    ?>
    <div class="poinzilla-order-sync-wrap">
      <div style="margin-bottom:12px">
        <label for="poinzilla_push_order">Clicca qui per inviare l'ordine manualmente:</label>
      </div>
      <input id="poinzilla_push_order" class="button button-primary" type="button" value="Sincronizza ora"
        data_id="<?php echo $meta_id->ID; ?>" />
      <div class="poinzilla-response"></div>
    </div>
    <?php
  }

  /**
   * Aggiunge gli script e gli stili necessari per la funzionalità di invio massivo
   */
  public function poinzilla_enqueue_bulk_scripts($hook) {
    // Verifica se siamo nella pagina degli ordini
    if ($hook != 'edit.php' && $hook != 'woocommerce_page_wc-orders') {
      return;
    }
    
    // Verifica se siamo nella pagina degli ordini di WooCommerce
    // Per l'interfaccia tradizionale, controlliamo post_type
    // Per HPS, controlliamo solo il valore di $hook che abbiamo già verificato
    if ($hook == 'edit.php') {
      global $post_type;
      if ($post_type != 'shop_order' && (!isset($_GET['post_type']) || $_GET['post_type'] != 'shop_order')) {
        return;
      }
    }
    // Se siamo qui e $hook è 'woocommerce_page_wc-orders', siamo nella pagina degli ordini HPS
    
    // Registra e carica lo script per la gestione dell'invio massivo
    wp_register_script(
      'poinzilla-bulk-sync',
      plugin_dir_url(dirname(__FILE__)) . 'admin/js/poinzilla-bulk-sync.js',
      array('jquery', 'jquery-ui-dialog'),
      POINZILLA_VERSION,
      true
    );
    
    // Ottieni gli stati degli ordini di WooCommerce
    $order_statuses = wc_get_order_statuses();
    
    // Passa i dati necessari allo script
    wp_localize_script('poinzilla-bulk-sync', 'poinzilla_bulk', array(
      'ajax_url' => admin_url('admin-ajax.php'),
      'nonce' => wp_create_nonce('poinzilla_bulk_nonce'),
      'processing_text' => __('Elaborazione in corso...', 'poinzilla'),
      'success_text' => __('Completato con successo', 'poinzilla'),
      'error_text' => __('Errore', 'poinzilla'),
      'close_text' => __('Chiudi', 'poinzilla'),
      'modal_title' => __('Invio ordini a Poinzilla', 'poinzilla'),
      'no_orders_selected' => __('Nessun ordine selezionato. Seleziona almeno un ordine.', 'poinzilla'),
      'too_many_orders' => __('Hai selezionato troppi ordini. Il limite massimo è %d ordini.', 'poinzilla'),
      'confirm_text' => __('Sei sicuro di voler inviare gli ordini selezionati a Poinzilla?', 'poinzilla'),
      'selected_orders_count' => __('Hai selezionato %d ordini', 'poinzilla'),
      'select_status' => __('Seleziona lo stato da inviare a Poinzilla (non modifica lo stato in WooCommerce):', 'poinzilla'),
      'yes_text' => __('Sì, procedi', 'poinzilla'),
      'no_text' => __('Annulla', 'poinzilla'),
      'order_statuses' => $order_statuses
    ));
    
    // Carica lo script
    wp_enqueue_script('poinzilla-bulk-sync');
    
    // Carica gli stili di jQuery UI Dialog e Dashicons
    wp_enqueue_style('wp-jquery-ui-dialog');
    wp_enqueue_style('dashicons');
  }
  
  /**
   * Aggiunge la modale per il feedback in tempo reale (WooCommerce 3.x)
   */
  public function poinzilla_add_custom_bulk_button_legacy($which) {
    global $post_type;
    
    // Verifica se siamo nella pagina degli ordini e nella posizione superiore
    if ($post_type == 'shop_order' && $which == 'top') {
      // Aggiungi solo la modale per il feedback in tempo reale
      $this->poinzilla_add_progress_modal();
    }
  }
  
  /**
   * Aggiunge la modale per il feedback in tempo reale (WooCommerce 4.x+)
   */
  public function poinzilla_add_custom_bulk_button() {
    // Aggiungi solo la modale per il feedback in tempo reale
    $this->poinzilla_add_progress_modal();
  }
  
  /**
   * Aggiunge la modale per mostrare il progresso dell'invio
   */
  public function poinzilla_add_progress_modal() {
    ?>
    <div id="poinzilla-progress-modal" style="display:none;">
      <div id="poinzilla-progress-container">
        <div id="poinzilla-progress-bar"></div>
        <div id="poinzilla-progress-status"></div>
        <div id="poinzilla-progress-log"></div>
      </div>
    </div>
    <?php
  }
  
  /**
   * Aggiunge la modale per mostrare il progresso dell'invio nell'interfaccia HPS
   */
  public function poinzilla_add_progress_modal_hps() {
    // Verifica se siamo nella pagina degli ordini con HPS abilitato
    global $pagenow, $hook_suffix;
    
    if ($hook_suffix === 'woocommerce_page_wc-orders' || ($pagenow === 'admin.php' && isset($_GET['page']) && $_GET['page'] === 'wc-orders')) {
      // Siamo nella pagina degli ordini con HPS, aggiungi il modale
      $this->poinzilla_add_progress_modal();
    }
  }
  
  /**
   * Gestisce la richiesta AJAX per processare gli ordini selezionati
   */
  public function poinzilla_bulk_process_orders_ajax() {
    // Verifica il nonce per la sicurezza
    check_ajax_referer('poinzilla_bulk_nonce', 'nonce');
    
    // Assicuriamoci che le classi necessarie siano caricate
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-logger.php';
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-order-sync.php';
    
    // Recupera gli ID degli ordini selezionati
    $order_ids = isset($_POST['order_ids']) ? array_map('intval', $_POST['order_ids']) : array();
    
    // Recupera lo stato dell'ordine selezionato
    $order_status = isset($_POST['order_status']) ? sanitize_text_field($_POST['order_status']) : '';
    
    // Verifica se ci sono ordini da processare
    if (empty($order_ids)) {
      wp_send_json_error(array(
        'message' => __('Nessun ordine selezionato.', 'poinzilla')
      ));
      return;
    }
    
    // Verifica se il numero di ordini supera il limite
    if (count($order_ids) > 10) {
      wp_send_json_error(array(
        'message' => __('Hai selezionato troppi ordini. Il limite massimo è 10 ordini.', 'poinzilla')
      ));
      return;
    }
    
    // Inizializza la classe di sincronizzazione
    $external_order = new Poinzilla_Sync_Order();
    $logger = new Poinzilla_Logger();
    
    // Inizializza i contatori
    $processed = 0;
    $failed = 0;
    $results = array();
    
    // Processa ogni ordine
    foreach ($order_ids as $order_id) {
      // Ottieni l'oggetto ordine
      $order = wc_get_order($order_id);
      
      if (!$order) {
        $failed++;
        $results[] = array(
          'id' => $order_id,
          'number' => $order_id,
          'status' => 'error',
          'message' => sprintf(__('Ordine #%s non trovato.', 'poinzilla'), $order_id)
        );
        continue;
      }
      
      $order_number = $order->get_order_number();
      
      // Logga lo stato selezionato (senza aggiornare l'ordine)
      if (!empty($order_status)) {
        $logger->debug(
          'Stato selezionato per l\'invio dell\'ordine #' . $order_number . ': ' . $order_status,
          array('source' => 'poinzilla-bulk-sync')
        );
      }
      
      // Prepara i dati da inviare, passando anche lo stato selezionato
      $payloadData = $external_order->poinzilla_woo_order_sync_data_prepare($order_id, $order_status);

      // Se non ci sono dati (ordine guest), salta questo ordine
      if ($payloadData === null) {
        $failed++;
        $results[] = array(
          'id' => $order_id,
          'number' => $order_number,
          'status' => 'skipped',
          'message' => sprintf(__('Ordine #%s saltato: ordine di un ospite (non registrato).', 'poinzilla'), $order_number)
        );
        continue;
      }

      // Effettua la chiamata API
      $response = $external_order->poinzilla_woo_order_sync_api($payloadData, get_option('poinzilla_enable_log'));
      
      // Gestisci la risposta
      if ($response['status'] == 200) {
        $processed++;
        $results[] = array(
          'id' => $order_id,
          'number' => $order_number,
          'status' => 'success',
          'message' => sprintf(__('Ordine #%s sincronizzato correttamente.', 'poinzilla'), $order_number)
        );
      } else {
        $failed++;
        $error_message = !empty($response['message']) ? $response['message'] : __('Errore sconosciuto.', 'poinzilla');
        
        $results[] = array(
          'id' => $order_id,
          'number' => $order_number,
          'status' => 'error',
          'message' => sprintf(__('Errore nell\'invio dell\'ordine #%s: %s', 'poinzilla'), $order_number, $error_message)
        );
        
        // Logga l'errore
        $logger->debug(
          'Errore invio ordine #' . $order_number . ': ' . $response['status'] . ' - ' . $error_message,
          array('source' => 'poinzilla-bulk-sync')
        );
      }
    }
    
    // Invia la risposta
    wp_send_json_success(array(
      'processed' => $processed,
      'failed' => $failed,
      'total' => count($order_ids),
      'results' => $results
    ));
  }

  /**
   * Gestisce la richiesta AJAX per l'invio manuale di un singolo ordine
   */
  public function poinzilla_push_external_order()
  {
    // Assicuriamoci che le classi necessarie siano caricate
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-logger.php';
    require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-order-sync.php';
    
    // Recupera l'ID dell'ordine dalla richiesta AJAX
    $resource_id = intval($_POST['order_id']);

    // Inizializza la classe di sincronizzazione
    $external_order = new Poinzilla_Sync_Order();

    // Prepara i dati da inviare (senza forzare uno stato specifico)
    $payloadData = $external_order->poinzilla_woo_order_sync_data_prepare($resource_id);

    // Se non ci sono dati (ordine guest), restituisci errore
    if ($payloadData === null) {
      wp_send_json_error(array(
        'message' => __('Impossibile inviare l\'ordine: ordine di un ospite (non registrato).', 'poinzilla')
      ));
      return;
    }

    // Effettua la chiamata API
    $response = $external_order->poinzilla_woo_order_sync_api($payloadData, get_option('poinzilla_enable_log'));

    // Definisce i messaggi di errore mappati
    $error_messages = array(
      400 => __('Dati inviati non validi. Verifica l\'ordine.', 'poinzilla'),
      401 => __('Non autorizzato. Controlla la chiave API.', 'poinzilla'),
      403 => __('Accesso negato. Verifica i permessi.', 'poinzilla'),
      404 => __('Risorsa non trovata. Controlla l\'ID dell\'ordine.', 'poinzilla'),
      500 => __('Errore interno del server Poinzilla. Riprova più tardi.', 'poinzilla'),
      503 => __('Servizio non disponibile. Riprova più tardi.', 'poinzilla'),
      'default' => __('Errore sconosciuto. Contatta il supporto.', 'poinzilla'),
    );

    // Gestisce la risposta
    if ($response['status'] == 200) {
      wp_send_json_success(array(
        'status' => 200,
        'message' => __('Ordine sincronizzato correttamente.', 'poinzilla'),
      ));
    } else {
      $status_code = $response['status'];
      $api_error_message = !empty($response['message']) ? $response['message'] : null;
      $error_message = isset($error_messages[$status_code]) ? $error_messages[$status_code] : $error_messages['default'];

      // Logga l'errore per il debug
      $logger = new Poinzilla_Logger();
      $logger->debug(
        'Errore API: ' . $status_code . ' - ' . $error_message . ($api_error_message ? ' (' . $api_error_message . ')' : ''),
        array('source' => 'poinzilla-order-sync')
      );

      // Invia il messaggio di errore al frontend
      wp_send_json_error(array(
        'status' => $status_code,
        'message' => $api_error_message ? $error_message . ' (' . $api_error_message . ')' : $error_message,
      ));
    }

    // Termina la richiesta
    wp_die();
  }
}