/**
 * Script per la gestione dell'invio massivo degli ordini a Poinzilla
 */
(function($) {
  'use strict';

  // Limite massimo di ordini selezionabili
  const MAX_ORDERS = 50;

  // Quando il documento è pronto
  $(document).ready(function() {
    // Inizializza la modale di progresso
    $('#poinzilla-progress-modal').dialog({
      autoOpen: false,
      modal: true,
      width: 650,
      height: 'auto',
      resizable: false,
      draggable: false,
      closeOnEscape: false,
      title: poinzilla_bulk.modal_title,
      dialogClass: 'poinzilla-dialog',
      buttons: [{
        text: poinzilla_bulk.close_text,
        class: 'button button-primary',
        click: function() {
          $(this).dialog('close');
        }
      }],
      open: function() {
        // Disabilita il pulsante di chiusura fino a quando il processo non è completato
        $(this).closest('.ui-dialog').find('.ui-dialog-buttonpane button').prop('disabled', true);
        
        // Sostituisci il testo "close" con una X
        var $closeButton = $(this).closest('.ui-dialog').find('.ui-dialog-titlebar-close');
        $closeButton.empty().text('×').css({
          'font-size': '24px',
          'font-weight': 'bold',
          'line-height': '24px',
          'color': '#fff',
          'text-align': 'center',
          'text-indent': '0'
        });
      },
      // Aggiungi animazione di apertura
      show: {
        effect: 'fadeIn',
        duration: 300
      },
      hide: {
        effect: 'fadeOut',
        duration: 200
      }
    });
    
    // Intercetta il click sui pulsanti "Applica" delle azioni di gruppo
    // Supporta sia l'interfaccia tradizionale che quella HPS
    $(document).on('click', '#doaction, #doaction2, .wc-orders-list-table .bulk-action-apply, .woocommerce-orders-table .bulk-action-apply', function(e) {
      // Ottieni il valore dell'azione selezionata
      var bulkAction;
      
      // Determina quale selettore usare in base all'elemento cliccato
      if ($(this).hasClass('bulk-action-apply')) {
        // Interfaccia HPS
        bulkAction = $(this).closest('.bulkactions').find('select').val();
      } else {
        // Interfaccia tradizionale
        bulkAction = $(this).prev('select').val();
      }
      
      // Se l'azione è "poinzilla_sync_orders", intercetta il click
      if (bulkAction === 'poinzilla_sync_orders') {
        // Previeni il comportamento predefinito (refresh della pagina)
        e.preventDefault();
        
        // Recupera gli ID degli ordini selezionati
        var orderIds = getSelectedOrderIds();
        
        // Verifica se ci sono ordini selezionati
        if (orderIds.length === 0) {
          showNotice(poinzilla_bulk.no_orders_selected, 'error');
          return;
        }
        
        // Verifica se il numero di ordini selezionati supera il limite
        if (orderIds.length > MAX_ORDERS) {
          showNotice(poinzilla_bulk.too_many_orders.replace('%d', MAX_ORDERS), 'error');
          return;
        }
        
        // Mostra la modale di conferma
        showConfirmDialog(orderIds);
      }
    });
  });

  /**
   * Mostra un avviso temporaneo
   */
  function showNotice(message, type) {
    // Rimuovi eventuali avvisi precedenti
    $('.poinzilla-notice-overlay').remove();
    
    // Crea l'elemento dell'avviso
    var notice = $('<div class="poinzilla-notice-overlay"><div class="poinzilla-notice ' + type + '">' + 
                  '<span class="dashicons dashicons-' + (type === 'error' ? 'warning' : 'yes-alt') + '"></span>' +
                  '<span class="message">' + message + '</span>' +
                  '<button type="button" class="close-notice"><span class="dashicons dashicons-no-alt"></span></button>' +
                  '</div></div>');
    
    // Aggiungi l'avviso al body
    $('body').append(notice);
    
    // Anima l'entrata dell'avviso
    setTimeout(function() {
      $('.poinzilla-notice').addClass('show');
    }, 10);
    
    // Aggiungi il gestore per la chiusura
    $('.close-notice').on('click', function() {
      $('.poinzilla-notice').removeClass('show');
      setTimeout(function() {
        $('.poinzilla-notice-overlay').remove();
      }, 300);
    });
    
    // Chiudi automaticamente dopo 4 secondi
    setTimeout(function() {
      if ($('.poinzilla-notice').length) {
        $('.poinzilla-notice').removeClass('show');
        setTimeout(function() {
          $('.poinzilla-notice-overlay').remove();
        }, 300);
      }
    }, 4000);
  }

  /**
   * Recupera gli ID degli ordini selezionati
   */
  function getSelectedOrderIds() {
    var orderIds = [];

    // Cerca le checkbox selezionate nella tabella degli ordini (supporta sia interfaccia tradizionale che HPS)
    // Prima prova il selettore per l'interfaccia tradizionale
    var $checkboxes = $('table.wp-list-table tbody input[type="checkbox"]:checked');
    
    // Se non trova checkbox con il selettore tradizionale, prova il selettore per HPS
    if ($checkboxes.length === 0) {
      $checkboxes = $('.wc-orders-list-table input[type="checkbox"]:checked, .woocommerce-orders-table input[type="checkbox"]:checked');
    }
    
    $checkboxes.each(function() {
      var orderId = $(this).val();
      if (orderId && orderId !== 'on') {
        orderIds.push(orderId);
      }
    });

    return orderIds;
  }

  /**
   * Mostra la modale di conferma
   */
  function showConfirmDialog(orderIds) {
    // Crea il contenuto della modale
    var dialogContent = $('<div class="poinzilla-confirm-dialog"></div>');
    
    // Aggiungi il messaggio di conferma
    dialogContent.append('<p class="poinzilla-confirm-message">' + poinzilla_bulk.confirm_text + '</p>');
    
    // Aggiungi il contatore degli ordini selezionati
    dialogContent.append('<p class="poinzilla-selected-count">' + 
      poinzilla_bulk.selected_orders_count.replace('%d', orderIds.length) + '</p>');
    
    // Aggiungi il selettore dello stato ordine
    var statusSelector = $('<div class="poinzilla-status-selector"></div>');
    statusSelector.append('<label for="poinzilla-order-status">' + poinzilla_bulk.select_status + '</label>');
    
    var statusSelect = $('<select id="poinzilla-order-status" name="order_status"></select>');
    
    // Aggiungi un'opzione vuota all'inizio
    statusSelect.append('<option value="">' + '-- Seleziona uno stato --' + '</option>');
    
    // Aggiungi le opzioni di stato
    $.each(poinzilla_bulk.order_statuses, function(value, label) {
      statusSelect.append('<option value="' + value + '">' + label + '</option>');
    });
    
    statusSelector.append(statusSelect);
    dialogContent.append(statusSelector);

    // Crea la modale
    dialogContent.dialog({
      modal: true,
      title: poinzilla_bulk.modal_title,
      width: 480,
      resizable: false,
      draggable: false,
      dialogClass: 'poinzilla-dialog',
      buttons: [
        {
          text: poinzilla_bulk.yes_text,
          class: 'button button-primary',
          click: function() {
            var selectedStatus = $('#poinzilla-order-status').val();
            $(this).dialog('close');
            processOrders(orderIds, selectedStatus);
          }
        },
        {
          text: poinzilla_bulk.no_text,
          class: 'button',
          click: function() {
            $(this).dialog('close');
          }
        }
      ],
      // Aggiungi animazione di apertura
      show: {
        effect: 'fadeIn',
        duration: 300
      },
      hide: {
        effect: 'fadeOut',
        duration: 200
      },
      // Quando la modale si apre
      open: function() {
        // Aggiungi animazione agli elementi interni
        var $dialog = $(this);
        $dialog.find('.poinzilla-confirm-message').css('opacity', 0).animate({opacity: 1}, 300);
        setTimeout(function() {
          $dialog.find('.poinzilla-selected-count').css('opacity', 0).animate({opacity: 1}, 300);
        }, 100);
        setTimeout(function() {
          $dialog.find('.poinzilla-status-selector').css('opacity', 0).animate({opacity: 1}, 300);
        }, 200);
        
        // Sostituisci il testo "close" con una X
        var $closeButton = $dialog.closest('.ui-dialog').find('.ui-dialog-titlebar-close');
        $closeButton.empty().text('×').css({
          'font-size': '24px',
          'font-weight': 'bold',
          'line-height': '24px',
          'color': '#fff',
          'text-align': 'center',
          'text-indent': '0'
        });
      }
    });
  }

  /**
   * Processa gli ordini selezionati
   */
  function processOrders(orderIds, orderStatus) {
    // Prepara la modale di progresso
    $('#poinzilla-progress-bar').html('<div class="progress" style="width: 0%"></div>');
    $('#poinzilla-progress-status').html(
      '<div class="poinzilla-status-icon processing"><span class="dashicons dashicons-update"></span></div>' +
      '<div class="poinzilla-status-text">' + poinzilla_bulk.processing_text + ' <span class="poinzilla-counter">0/' + orderIds.length + '</span></div>'
    );
    $('#poinzilla-progress-log').empty();

    // Apre la modale di progresso
    $('#poinzilla-progress-modal').dialog('open');

    // Anima la barra di progresso iniziale
    setTimeout(function() {
      $('#poinzilla-progress-bar .progress').css({
        'width': '5%',
        'transition': 'width 0.5s cubic-bezier(0.4, 0, 0.2, 1)'
      });
    }, 300);

    // Inizializza i contatori
    var processed = 0;
    var failed = 0;
    var results = [];
    
    // Funzione per processare un singolo ordine
    function processOrder(index) {
      if (index >= orderIds.length) {
        // Tutti gli ordini sono stati processati
        completeProcess();
        return;
      }
      
      var orderId = orderIds[index];
      var progressPercent = 5 + Math.round((index / orderIds.length) * 90); // Da 5% a 95%
      
      // Aggiorna la barra di progresso
      $('#poinzilla-progress-bar .progress').css('width', progressPercent + '%');
      
      // Aggiorna il contatore
      $('.poinzilla-counter').text(index + '/' + orderIds.length);
      
      // Effettua la chiamata AJAX per un singolo ordine
      $.ajax({
        url: poinzilla_bulk.ajax_url,
        type: 'POST',
        data: {
          action: 'poinzilla_bulk_process_orders',
          order_ids: [orderId], // Invia un solo ordine alla volta
          order_status: orderStatus,
          nonce: poinzilla_bulk.nonce
        },
        success: function(response) {
          if (response.success) {
            // Aggiorna i contatori
            processed += response.data.processed;
            failed += response.data.failed;
            
            // Aggiungi i risultati all'array
            results = results.concat(response.data.results);
            
            // Aggiungi l'elemento al log immediatamente
            var result = response.data.results[0]; // Dovrebbe esserci un solo risultato
            addLogItem(result);
            
            // Processa il prossimo ordine
            setTimeout(function() {
              processOrder(index + 1);
            }, 3000);
          } else {
            // Gestisce l'errore
            failed++;
            var errorResult = {
              id: orderId,
              status: 'error',
              message: response.data.message || poinzilla_bulk.error_text
            };
            results.push(errorResult);
            addLogItem(errorResult);
            
            // Processa il prossimo ordine
            setTimeout(function() {
              processOrder(index + 1);
            }, 3000);
          }
        },
        error: function(xhr, status, error) {
          // Gestisce l'errore di connessione
          failed++;
          var errorResult = {
            id: orderId,
            status: 'error',
            message: error || poinzilla_bulk.error_text
          };
          results.push(errorResult);
          addLogItem(errorResult);
          
          // Processa il prossimo ordine
          setTimeout(function() {
            processOrder(index + 1);
          }, 3000);
        }
      });
    }
    
    // Funzione per aggiungere un elemento al log
    function addLogItem(result) {
      var logClass = result.status === 'success' ? 'success' : 'error';
      var logIcon = result.status === 'success' ? 
        '<span class="dashicons dashicons-yes-alt"></span>' : 
        '<span class="dashicons dashicons-warning"></span>';
        
      var $item = $(
        '<div class="poinzilla-log-item ' + logClass + '" style="opacity:0">' + 
          '<div class="poinzilla-log-icon">' + logIcon + '</div>' +
          '<div class="poinzilla-log-message">' + result.message + '</div>' +
        '</div>'
      );
      
      $('#poinzilla-progress-log').append($item);
      
      // Anima l'entrata dell'elemento
      setTimeout(function() {
        $item.animate({opacity: 1}, 300);
        
        // Scorri verso il basso per mostrare l'elemento appena aggiunto
        $('#poinzilla-progress-log').scrollTop($('#poinzilla-progress-log')[0].scrollHeight);
      }, 50);
    }
    
    // Funzione per completare il processo
    function completeProcess() {
      // Aggiorna la barra di progresso al 100%
      $('#poinzilla-progress-bar .progress').css('width', '100%');
      
      // Aggiorna lo stato
      var statusIcon = failed > 0 ? 
        '<div class="poinzilla-status-icon warning"><span class="dashicons dashicons-warning"></span></div>' :
        '<div class="poinzilla-status-icon success"><span class="dashicons dashicons-yes-alt"></span></div>';
        
      var statusText = poinzilla_bulk.success_text + ': ' + 
                      processed + '/' + orderIds.length + ' ' + 
                      (failed > 0 ? '(' + failed + ' ' + poinzilla_bulk.error_text + ')' : '');
      
      $('#poinzilla-progress-status').html(statusIcon + '<div class="poinzilla-status-text">' + statusText + '</div>');
      
      // Abilita il pulsante di chiusura
      $('#poinzilla-progress-modal').closest('.ui-dialog').find('.ui-dialog-buttonpane button').prop('disabled', false);
    }
    
    // Inizia a processare gli ordini
    setTimeout(function() {
      processOrder(0);
    }, 3000);
  }

  // La funzione populateLogWithAnimation è stata rimossa perché ora aggiungiamo gli elementi al log in tempo reale

  // Aggiungi stili CSS per le notifiche
  $('<style>')
    .prop('type', 'text/css')
    .html(`
      .poinzilla-notice-overlay {
        position: fixed;
        top: 32px;
        right: 20px;
        z-index: 9999;
      }
      .poinzilla-notice {
        background: white;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        padding: 12px 16px;
        display: flex;
        align-items: center;
        gap: 12px;
        max-width: 400px;
        transform: translateY(-20px);
        opacity: 0;
        transition: all 0.3s ease;
      }
      .poinzilla-notice.show {
        transform: translateY(0);
        opacity: 1;
      }
      .poinzilla-notice.error {
        border-left: 4px solid #FC573B;
      }
      .poinzilla-notice.success {
        border-left: 4px solid #188F86;
      }
      .poinzilla-notice .dashicons-warning {
        color: #FC573B;
        font-size: 24px;
      }
      .poinzilla-notice .dashicons-yes-alt {
        color: #188F86;
        font-size: 24px;
      }
      .poinzilla-notice .message {
        flex: 1;
        font-size: 14px;
      }
      .poinzilla-notice .close-notice {
        background: none;
        border: none;
        cursor: pointer;
        color: #999;
        padding: 0;
        display: flex;
        align-items: center;
        justify-content: center;
      }
      .poinzilla-notice .close-notice:hover {
        color: #333;
      }
    `)
    .appendTo('head');

})(jQuery);