<?php
defined('ABSPATH') or exit;

/**
 * Fired during plugin activation
 *
 * @link       https://www.zoorate.com
 * @since      1.0.0
 *
 * @package    Poinzilla
 * @subpackage Poinzilla/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Poinzilla
 * @subpackage Poinzilla/includes
 * @author     Zoorate srl <info@zoorate.com>
 */
class Poinzilla_Sync_Order
{


  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct() {}

  public function poinzilla_woo_order_sync_data_prepare($id, $custom_status = '')
  {
    // $data = WC()->api->get_endpoint_data("/wc/v3/orders/{$id}");

    $order = wc_get_order($id);

    // Ottieni l'email dell'utente registrato
    $customer_email = '';
    $customer_id = $order->get_customer_id();
    if ($customer_id > 0) {
      $customer = get_user_by('id', $customer_id);
      if ($customer) {
        $customer_email = $customer->user_email;
      }
    }

    // Se non c'è un utente registrato, non inviare l'ordine
    if (empty($customer_email)) {
      return null;
    }

    $payload_data = array(
      "id" => (string) $id,
      "customer_id" => (string) $customer_id,
      "customer_email" => $customer_email,
      "line_items" => array(),
    );

    // Se è stato fornito uno stato personalizzato, usalo, altrimenti usa lo stato attuale dell'ordine
    if (!empty($custom_status)) {
      $payload_data['status'] = $custom_status;
    } else {
      $payload_data['status'] = 'wc-' . $order->get_status();
    }
    // print_r($data['line_items']);
    $total = 0;
    foreach ($order->get_items() as $key => $itemdata) {
      $itemdata = $itemdata->get_data();
      $lineItem = array(
        "id" => (string) $itemdata['product_id'],
        "product_id" => (string) $itemdata['product_id'],
        "productCat" => array(),
        "total" => number_format((float) ($itemdata['subtotal'] + $itemdata['subtotal_tax']), 2, '.', '')
      );

      $product_cat_ids = wp_get_object_terms($itemdata['product_id'], 'product_cat', array('fields' => 'ids'));

      $all_cat_ids = array();
      foreach ($product_cat_ids as $cat_id) {
        $all_cat_ids[] = $cat_id;
        $parent_ids = get_ancestors($cat_id, 'product_cat', 'taxonomy');
        $all_cat_ids = array_merge($all_cat_ids, $parent_ids);
      }
      $all_cat_ids = array_unique((array) $all_cat_ids);
      $lineItem['productCat'] = array_map('strval', array_values($all_cat_ids));

      $total = $total + $itemdata['subtotal'] + $itemdata['subtotal_tax'];
      $payload_data['line_items'][] = $lineItem;
    }
    if (defined('ICL_SITEPRESS_VERSION')) :
      $default_lang = apply_filters('wpml_default_language', NULL);
      $product_array =  $payload_data['line_items'];
      $payload_data['line_items'] = array();
      $languages = apply_filters('wpml_active_languages', NULL, array('skip_missing' => 0));
      if (!empty($languages)):
        foreach ($languages as $lang) {
          if ($lang['code'] == $default_lang):
            foreach ($product_array as $lindata) {
              $translated_id = apply_filters('wpml_object_id', $lindata['id'], 'product', FALSE, $lang['code']);
              $lindata['id'] = (string) $translated_id;
              $lindata['product_id'] = (string) $translated_id;
              foreach ($lindata['productCat'] as $key => $catid) {
                $lindata['productCat'][$key] = strval(apply_filters('wpml_object_id', $catid, 'product_cat', FALSE, $lang['code']));
              }
              $payload_data['line_items'][] = $lindata;
            }
          endif;
        }
      endif;
    endif;

    $payload_data["total"] = number_format((float) $total, 2, '.', '');
    foreach ($order->get_coupons() as $coupon) {
      $coupon = $coupon->get_data();
      $couponData = array(
        "id" => (string) $coupon['id'],
        "code" => $coupon['code'],
        "discount" => number_format((float) ($coupon['discount'] + $coupon['discount_tax']), 2, '.', '')
      );
      $payload_data['coupon_lines'][] = $couponData;
    }

    return $payload_data;
  }
  public function poinzilla_woo_order_sync_api($payload_data, $is_log)
  {
    require_once plugin_dir_path(__FILE__) . 'class-poinzilla-logger.php';

    $logger = new Poinzilla_Logger();
    $logger->debug(
      'order sync API call data: ' . json_encode($payload_data, JSON_PRETTY_PRINT),
      array('source' => 'poinzilla-order-sync')
    );
    $response = wp_remote_post(
      POINZILLA_API_URL_V1 . "/api/External/Order",
      array(
        'method' => 'POST',
        'timeout' => 45,
        'redirection' => 5,
        'httpversion' => '1.0',
        'blocking' => true,
        'headers' => array(
          'X-loyalty-channel-key' => get_option('poinzilla_api_key'),
          'Content-type' => 'application/json',
        ),
        'body' => json_encode($payload_data),
        'cookies' => array(),
      )
    );

    // Logga la risposta cruda per debug
    $logger->debug('Raw API Response: ' . print_r($response, true), array('source' => 'poinzilla-order-sync'));

    // Analizza lo stato della risposta
    if (is_wp_error($response)) {
      $error_message = $response->get_error_message();
      $logger->debug('API Call Failed: ' . $error_message, array('source' => 'poinzilla-order-sync'));

      return array(
        'status' => 500,
        'message' => __('Errore di connessione con l\'API.', 'poinzilla'),
      );
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);

    // Gestione dei diversi status
    if ($response_code == 200) {
      $logger->debug('Poinzilla risposta:' . $response_body, array('source' => 'poinzilla-order-sync'));
      return array(
        'status' => 200,
        'message' => __('Ordine sincronizzato correttamente.', 'poinzilla'),
      );
    } elseif ($response_code == 400) {
      $logger->debug('Poinzilla risposta: ' . $response_body, array('source' => 'poinzilla-order-sync'));
      return array(
        'status' => 400,
        'message' => __('Errore nei dati inviati all\'API.', 'poinzilla'),
      );
    } elseif ($response_code == 500) {
      $logger->debug('Poinzilla risposta:' . $response_body, array('source' => 'poinzilla-order-sync'));
      return array(
        'status' => 500,
        'message' => __('Errore interno al server Poinzilla. Contata il servizio clienti.', 'poinzilla'),
      );
    } else {
      $logger->debug('Poinzilla risposta:' . $response_body, array('source' => 'poinzilla-order-sync'));
      return array(
        'status' => $response_code,
        'message' => __('Risposta inattesa dall\'API.', 'poinzilla'),
      );
    }
  }
  public function poinzilla_woo_order_status_change($id, $previous_status, $next_status)
  {
    $setup_mode = get_option('poinzilla_setup_mode');

    if ($setup_mode) {
      return;
    }
    $order = wc_get_order($id);

    if (!$order) {
      return;
    }
    // if ($order->get_meta('_poinzilla_order_sync')) {
    //   return;
    // }
    $payload_data = $this->poinzilla_woo_order_sync_data_prepare($id);
    if ($payload_data !== null) {
      $this->poinzilla_woo_order_sync_api($payload_data, get_option('poinzilla_enable_log'));
    }
  }
}
