<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://www.zoorate.com
 * @since      1.0.0
 *
 * @package    Poinzilla
 * @subpackage Poinzilla/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Poinzilla
 * @subpackage Poinzilla/includes
 * @author     Zoorate srl <info@zoorate.com>
 */
class Poinzilla
{

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Poinzilla_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct()
	{
		global $poizilla_is_setup_mode, $poizilla_is_enable_log;
		$poizilla_is_setup_mode = false;
		$poizilla_is_enable_log = false;
		if (get_option('poinzilla_setup_mode')) {
			$poizilla_is_setup_mode = true;
		}
		if (get_option('poinzilla_enable_log')) {
			$poizilla_is_enable_log = true;
		}
		if (defined('POINZILLA_VERSION')) {
			$this->version = POINZILLA_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'poinzilla';
		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_send_order_hook_class();
		$this->define_rewrite_rule();
		$this->define_users_export_class();
		$this->poinzilla_add_meta_box();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Poinzilla_Loader. Orchestrates the hooks of the plugin.
	 * - Poinzilla_i18n. Defines internationalization functionality.
	 * - Poinzilla_Admin. Defines all hooks for the admin area.
	 * - Poinzilla_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies()
	{

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-poinzilla-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-poinzilla-public.php';


		/**
		 * The class responsible for consumer group related action
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-consumer-group.php';


		/**
		 * The class responsible for defining all webhook related action
		 * side of the site.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-order-sync.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-rewrite-rule.php';

		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-users-export.php';

		require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-poinzilla-push-order.php';

		$this->loader = new Poinzilla_Loader();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Poinzilla_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale()
	{

		$plugin_i18n = new Poinzilla_i18n();

		$this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks()
	{

		$plugin_admin = new Poinzilla_Admin($this->get_plugin_name(), $this->get_version());


		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
		$this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');

		/**
		 *add_menu_page_function 
		 */
		$this->loader->add_action('admin_menu', $plugin_admin, 'poinzilla_admin_menu');

		$this->loader->add_action('poinzilla_activated', $plugin_admin, 'poinzilla_woocommerce_notice');
		$this->loader->add_action('admin_notices', $plugin_admin, 'poinzilla_woocommerce_notice');
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks()
	{

		$plugin_public = new Poinzilla_Public($this->get_plugin_name(), $this->get_version());




		$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'poinzilla_enqueue_styles');
		$this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'poinzilla_enqueue_scripts');


		$this->loader->add_action('wp_footer', $plugin_public, 'poinzilla_footer_hook');
		$this->loader->add_action('profile_update', $plugin_public, 'poinzilla_user_register', 100000, 3);

		$this->loader->add_action('wp_update_user', $plugin_public, 'poinzilla_user_register', 100000, 3);


		// $this->loader->add_action('woocommerce_rest_api_get_rest_namespaces', $this, 'poinzilla_register_rest_namespaces', 10, 1);
		$this->loader->add_action('rest_api_init', $this, 'poinzilla_register_rest_namespaces', 10, 1);

		//add support for administration rols
		$this->loader->add_filter('woocommerce_rest_customer_allowed_roles', $this, 'poinzilla_add_administration_to_allowed_roles', 10, 1);
		$this->loader->add_action('woocommerce_rest_insert_customer', $this, 'poinzilla_add_role_to_user', 10, 2);
	}



	public function poinzilla_register_rest_namespaces($namespaces)
	{
		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-rest.php';
		$controller = new WC_API_Poinzilla();
		$controller->register_routes();

		require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-poinzilla-consumer-group.php';
		$consumer_controller = new Poinzilla_Consumer_Group();
		$consumer_controller->register_roles_endpoint();

		return $namespaces;
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_send_order_hook_class()
	{

		$plugin_public = new Poinzilla_Sync_Order();

		$this->loader->add_action('woocommerce_order_status_changed', $plugin_public, 'poinzilla_woo_order_status_change', 10, 3);
	}

	private function poinzilla_add_meta_box()
	{
		// Inizializza la classe per la gestione degli ordini
		$poinzilla_pushorder = new Poinzilla_PushOrder();

		// Registra la metabox per l'invio manuale degli ordini
		$this->loader->add_action('add_meta_boxes', $poinzilla_pushorder, 'poinzilla_register_meta_box');
		
		// Gestisce la richiesta AJAX per l'invio manuale degli ordini
		$this->loader->add_action('wp_ajax_poinzilla_push_external_order', $poinzilla_pushorder, 'poinzilla_push_external_order');
		
		// Nota: I filtri per le bulk actions sono già registrati nel costruttore della classe Poinzilla_PushOrder
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_rewrite_rule()
	{
		$plugin_public = new Poinzilla_Rewrite_Rule();
		$this->loader->add_action('init', $plugin_public, 'poinzilla_add_rewrite_rule');
		$this->loader->add_filter('query_vars', $plugin_public, 'poinzilla_add_query_vars');
		$this->loader->add_filter('template_include', $plugin_public, 'poinzilla_template_include');
	}


	private function define_users_export_class()
	{
		$this->user_export = new Poinzilla_User_Export();
	}

	public function poinzilla_add_administration_to_allowed_roles($roles)
	{
		$roles[] = 'administration';
		return $roles;
	}
	public function poinzilla_add_role_to_user($user_data, $request)
	{
		$role = $request->get_param('role');
		if ($role):
			$user_obj = new WP_User($user_data->ID);
			$user_obj->add_role($role);
		endif;
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run()
	{
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name()
	{
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Poinzilla_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader()
	{
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version()
	{
		return $this->version;
	}
}
